<?php
require_once __DIR__ . '/session_init.php';

use Parking\Auth;
use Parking\Database;
use Parking\Config;
use Parking\ParkingRepository;
use Parking\EventRepository;
use Parking\Installer;

$container = require __DIR__ . '/../bootstrap.php';
$config = $container['config'];
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));
if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

$auth = new Auth($database);
$auth->requireAuth();

$isAdmin = true;
$userFullName = $auth->getAdminFullName();

$repository = new ParkingRepository($database);
$eventRepo = new EventRepository($database);
$activeEvent = $eventRepo->getActiveEvent();

// Get statistics
$pdo = $database->connection();

$totalStmt = $pdo->query("SELECT COUNT(*) as total FROM tickets WHERE status != 'returned'");
$total = $totalStmt->fetch()['total'] ?? 0;

$parkedStmt = $pdo->query("SELECT COUNT(*) as total FROM tickets WHERE status = 'parked'");
$parked = $parkedStmt->fetch()['total'] ?? 0;

$requestedStmt = $pdo->query("SELECT COUNT(*) as total FROM tickets WHERE status = 'requested'");
$requested = $requestedStmt->fetch()['total'] ?? 0;

// Returned today: use driver-specific date function
$driver = $pdo->getAttribute(PDO::ATTR_DRIVER_NAME);
if ($driver === 'mysql') {
    $returnedTodayStmt = $pdo->query("SELECT COUNT(*) as total FROM tickets WHERE status = 'returned' AND DATE(returned_at) = CURDATE()");
} else {
    $returnedTodayStmt = $pdo->query("SELECT COUNT(*) as total FROM tickets WHERE status = 'returned' AND DATE(returned_at) = DATE('now')");
}
$returnedToday = $returnedTodayStmt->fetch()['total'] ?? 0;

// Get recent tickets
$recentStmt = $pdo->query("SELECT * FROM tickets ORDER BY created_at DESC LIMIT 10");
$recentTickets = $recentStmt->fetchAll();

// Get recent activity (last 5 actions)
$activityStmt = $pdo->query("
    SELECT 
        customer_name,
        license_plate,
        status,
        created_at,
        requested_at,
        returned_at
    FROM tickets 
    ORDER BY 
        CASE 
            WHEN returned_at IS NOT NULL THEN returned_at
            WHEN requested_at IS NOT NULL THEN requested_at
            ELSE created_at
        END DESC
    LIMIT 5
");
$activities = $activityStmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لوحة التحكم - نبيه أوتو</title>
    <!-- Material Icons -->
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/admin-dashboard.css">
</head>
<body class="sidebar-fixed">
    <?php 
    $currentPage = 'admin-dashboard';
    include __DIR__ . '/includes/sidebar.php'; 
    ?>

    <!-- Main Content -->
    <main class="main-content">
        <!-- Top Bar -->
        <div class="topbar">
            <div>
                <button class="btn" id="sidebarToggle" aria-label="القائمة" style="display:none;">
                    <span class="material-icons">menu</span>
                </button>
                <h1 class="page-title">مرحباً، <?php echo htmlspecialchars($auth->getAdminFullName() ?? 'Admin', ENT_QUOTES, 'UTF-8'); ?>!</h1>
                <p class="page-subtitle">إليك ملخص اليوم لنظام المواقف</p>
            </div>
            <div class="topbar-actions">
                <button class="btn btn-primary" onclick="window.location.href='register-vehicle.php'">
                    <span class="material-icons" style="font-size: 18px;">add</span>
                    تسجيل سيارة جديدة
                </button>
            </div>
        </div>

        <!-- Event Banner -->
        <?php if ($activeEvent): ?>
            <div style="background: linear-gradient(135deg, rgba(108, 99, 255, 0.08) 0%, rgba(0, 217, 166, 0.06) 100%); border: 1px solid rgba(108, 99, 255, 0.15); border-radius: 16px; padding: 16px 24px; margin-bottom: 24px; display: flex; align-items: center; gap: 20px;">
                <?php if (!empty($activeEvent['logo'])): ?>
                    <div style="flex-shrink: 0;">
                        <img src="<?php echo htmlspecialchars($activeEvent['logo']); ?>" alt="Event Logo" style="height: 60px; width: auto; max-width: 120px; object-fit: contain; filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));" />
                    </div>
                <?php endif; ?>
                
                <div style="flex: 1; min-width: 0;">
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 6px;">
                        <span class="material-icons" style="font-size: 20px; color: var(--primary);">event</span>
                        <h3 style="font-size: 20px; font-weight: 700; color: var(--primary); margin: 0;">
                            <?php echo htmlspecialchars($activeEvent['name']); ?>
                        </h3>
                    </div>
                    
                    <div style="display: flex; flex-wrap: wrap; gap: 16px; font-size: 13px; color: var(--gray);">
                        <?php if ($activeEvent['event_date']): ?>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--primary);">calendar_today</span>
                                <span><?php echo htmlspecialchars($activeEvent['event_date']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($activeEvent['event_location']): ?>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--success);">location_on</span>
                                <span><?php echo htmlspecialchars($activeEvent['event_location']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($activeEvent['contact_phone']): ?>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--warning);">phone</span>
                                <span><?php echo htmlspecialchars($activeEvent['contact_phone']); ?></span>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Stats Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon primary">
                        <span class="material-icons">directions_car</span>
                    </div>
                </div>
                <div class="stat-title">إجمالي السيارات النشطة</div>
                <div class="stat-value"><?php echo $total; ?></div>
                <div class="stat-progress">
                    <div class="stat-progress-info">
                        <span class="stat-progress-label">نسبة الإشغال</span>
                        <span class="stat-progress-value"><?php echo $total > 0 ? '100%' : '0%'; ?></span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar" style="width: <?php echo $total > 0 ? '100' : '0'; ?>%"></div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon success">
                        <span class="material-icons">check_circle</span>
                    </div>
                </div>
                <div class="stat-title">تم إنجاز التوقيف</div>
                <div class="stat-value"><?php echo $parked; ?></div>
                <div class="stat-progress">
                    <div class="stat-progress-info">
                        <span class="stat-progress-label"><?php echo $total > 0 ? round(($parked / $total) * 100, 1) : 0; ?>% من إجمالي السيارات متوقفة</span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar" style="width: <?php echo $total > 0 ? round(($parked / $total) * 100) : 0; ?>%"></div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon warning">
                        <span class="material-icons">assignment</span>
                    </div>
                </div>
                <div class="stat-title">الأدوار المسندة</div>
                <div class="stat-value"><?php echo $requested; ?></div>
                <div class="stat-progress">
                    <div class="stat-progress-info">
                        <span class="stat-progress-label"><?php echo $total > 0 ? round(($requested / $total) * 100, 1) : 0; ?>% من السيارات قيد التجهيز</span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar" style="width: <?php echo $total > 0 ? round(($requested / $total) * 100) : 0; ?>%"></div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Global Statistics Section -->
        <h2 style="color: var(--light); font-size: 24px; margin-bottom: 20px; font-weight: 700;">الإحصائيات العامة</h2>
        
        <div class="stats-grid" style="grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));">
            <div class="stat-card">
                <div class="stat-title">إجمالي العملاء</div>
                <div class="stat-value"><?php echo $total; ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-title">تم إكمال التوقيف</div>
                <div class="stat-value"><?php echo $returnedToday; ?></div>
                <div class="stat-progress">
                    <div class="stat-progress-info">
                        <span class="stat-progress-label"><?php echo $total > 0 ? round(($returnedToday / ($total + $returnedToday)) * 100, 1) : 0; ?>% من إجمالي العملاء أكملوا التوقيف</span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar" style="width: <?php echo $total > 0 ? round(($returnedToday / ($total + $returnedToday)) * 100) : 0; ?>%"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Content Grid -->
        <div class="content-grid">
            <!-- Recent Tickets Table -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">السيارات الأخيرة</h2>
                    <div class="card-actions">
                        <button class="btn btn-primary" onclick="window.location.href='dashboard.php'">
                            عرض الكل
                        </button>
                    </div>
                </div>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>العميل</th>
                                <th>رقم اللوحة</th>
                                <th>المنسق</th>
                                <th>الحالة</th>
                                <th>الوقت</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($recentTickets)): ?>
                                <tr>
                                    <td colspan="6" style="text-align: center; padding: 40px; color: var(--gray);">
                                        <span class="material-icons" style="font-size: 48px; opacity: 0.3;">inbox</span>
                                        <br>لا توجد سيارات مسجلة حالياً
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($recentTickets as $ticket): ?>
                                    <tr>
                                        <td><strong>#<?php echo $ticket['id']; ?></strong></td>
                                        <td><?php echo htmlspecialchars($ticket['customer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td><strong><?php echo htmlspecialchars($ticket['license_plate'], ENT_QUOTES, 'UTF-8'); ?></strong></td>
                                        <td><?php echo htmlspecialchars($ticket['valet_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td>
                                            <?php
                                            $statusLabels = [
                                                'parked' => ['موقف', 'info'],
                                                'requested' => ['قيد التجهيز', 'warning'],
                                                'returned' => ['تم التسليم', 'success']
                                            ];
                                            $status = $statusLabels[$ticket['status']] ?? ['غير معروف', 'danger'];
                                            ?>
                                            <span class="badge <?php echo $status[1]; ?>">
                                                <?php echo $status[0]; ?>
                                            </span>
                                        </td>
                                        <td style="font-size: 12px; color: var(--gray);">
                                            <?php 
                                            $time = strtotime($ticket['created_at']);
                                            echo date('H:i', $time);
                                            ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Recent Activity -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">النشاط الأخير</h2>
                </div>
                <div>
                    <?php if (empty($activities)): ?>
                        <p style="text-align: center; padding: 40px; color: var(--gray);">
                            <span class="material-icons" style="font-size: 48px; opacity: 0.3;">inbox</span>
                            <br>لا يوجد نشاط حالياً
                        </p>
                    <?php else: ?>
                        <?php foreach ($activities as $activity): ?>
                            <?php
                            $icons = [
                                'parked' => ['local_parking', 'info'],
                                'requested' => ['schedule', 'warning'],
                                'returned' => ['check_circle', 'success']
                            ];
                            $icon = $icons[$activity['status']] ?? ['description', 'info'];
                            
                            $actions = [
                                'parked' => 'تم تسجيل السيارة',
                                'requested' => 'طلب التجهيز',
                                'returned' => 'تم التسليم'
                            ];
                            $actionText = $actions[$activity['status']] ?? 'نشاط';
                            ?>
                            <div class="activity-item">
                                <div class="activity-icon <?php echo $icon[1]; ?>" style="background: rgba(99, 102, 241, 0.1); color: var(--primary);">
                                    <span class="material-icons"><?php echo $icon[0]; ?></span>
                                </div>
                                <div class="activity-content">
                                    <div class="activity-title"><?php echo $actionText; ?></div>
                                    <div class="activity-desc">
                                        <?php echo htmlspecialchars($activity['customer_name'], ENT_QUOTES, 'UTF-8'); ?> - 
                                        <?php echo htmlspecialchars($activity['license_plate'], ENT_QUOTES, 'UTF-8'); ?>
                                    </div>
                                    <div class="activity-time">
                                        <?php 
                                        $activityTime = strtotime($activity['returned_at'] ?? $activity['requested_at'] ?? $activity['created_at']);
                                        $diff = time() - $activityTime;
                                        if ($diff < 60) {
                                            echo 'منذ لحظات';
                                        } elseif ($diff < 3600) {
                                            echo 'منذ ' . floor($diff / 60) . ' دقيقة';
                                        } else {
                                            echo 'منذ ' . floor($diff / 3600) . ' ساعة';
                                        }
                                        ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <div class="sidebar-backdrop" id="sidebarBackdrop"></div>
    
    <script src="assets/sidebar.js"></script>
    <script>
        // Auto refresh every 30 seconds
        setInterval(() => {
            location.reload();
        }, 30000);

        // Add smooth animations
        document.addEventListener('DOMContentLoaded', () => {
            const cards = document.querySelectorAll('.stat-card, .card');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                setTimeout(() => {
                    card.style.transition = 'all 0.5s ease-out';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
    </script>
</body>
</html>
