<?php

declare(strict_types=1);

use Parking\Installer;
use Parking\ParkingRepository;
use Parking\Notifier;
use Parking\Config;

require_once __DIR__ . '/../session_init.php';

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Allow: POST');
    http_response_code(405);
    echo json_encode(['error' => 'طريقة غير مدعومة.'], JSON_UNESCAPED_UNICODE);
    exit;
}

// Authentication check - require admin or coordinator session
if (!isset($_SESSION['admin_id']) && !isset($_SESSION['coordinator_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'غير مصرح بالوصول.'], JSON_UNESCAPED_UNICODE);
    exit;
}

$container = require __DIR__ . '/../../bootstrap.php';
$database = $container['database'];
$config = $container['config'];

$installer = new Installer(dirname(__DIR__, 2));

if (!$installer->isInstalled($database)) {
    http_response_code(503);
    echo json_encode(['error' => 'التطبيق غير مُعد بعد.'], JSON_UNESCAPED_UNICODE);
    exit;
}

$input = json_decode(file_get_contents('php://input') ?: '', true);

if (!is_array($input)) {
    $input = $_POST;
}

$ticketId = isset($input['ticket_id']) ? (int) $input['ticket_id'] : 0;
$status = isset($input['status']) ? trim((string) $input['status']) : '';

if ($ticketId <= 0 || $status === '') {
    http_response_code(422);
    echo json_encode(['error' => 'البيانات غير مكتملة.'], JSON_UNESCAPED_UNICODE);
    exit;
}

// Initialize Notifier for SMS
$notifier = null;
try {
    $notifier = new Notifier($config);
} catch (Throwable $e) {
    // SMS not configured, continue without it
    error_log("Notifier initialization failed: " . $e->getMessage());
}

$repository = new ParkingRepository($database, $notifier);

try {
    $ticket = $repository->setStatus($ticketId, $status);
} catch (Throwable $exception) {
    http_response_code(400);
    echo json_encode(['error' => $exception->getMessage()], JSON_UNESCAPED_UNICODE);
    exit;
}

if (!$ticket) {
    http_response_code(404);
    echo json_encode(['error' => 'لم يتم العثور على التذكرة.'], JSON_UNESCAPED_UNICODE);
    exit;
}

echo json_encode([
    'ticket' => $ticket,
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
