<?php
require_once __DIR__ . '/session_init.php';

use Parking\Auth;
use Parking\Database;
use Parking\CoordinatorRepository;
use Parking\EventRepository;
use Parking\Installer;

$container = require __DIR__ . '/../bootstrap.php';
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));
if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

$auth = new Auth($database);
$auth->requireAuth();

$isAdmin = true;
$userFullName = $auth->getAdminFullName();

$coordinatorRepo = new CoordinatorRepository($database);
$eventRepo = new EventRepository($database);
$activeEvent = $eventRepo->getActiveEvent();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create':
            $coordinatorRepo->createCoordinator([
                'event_id' => $_POST['event_id'],
                'full_name' => $_POST['full_name'],
                'username' => $_POST['username'],
                'password' => $_POST['password'],
                'phone' => $_POST['phone'],
                'email' => $_POST['email']
            ]);
            header('Location: coordinators.php?success=created');
            exit;
            
        case 'update':
            $coordinatorRepo->updateCoordinator($_POST['coordinator_id'], [
                'event_id' => $_POST['event_id'],
                'full_name' => $_POST['full_name'],
                'username' => $_POST['username'],
                'password' => $_POST['password'], // Will be empty if not changed
                'phone' => $_POST['phone'],
                'email' => $_POST['email']
            ]);
            header('Location: coordinators.php?success=updated');
            exit;
            
        case 'toggle':
            $coordinatorRepo->toggleCoordinatorStatus($_POST['coordinator_id']);
            header('Location: coordinators.php?success=toggled');
            exit;
            
        case 'delete':
            $coordinatorRepo->deleteCoordinator($_POST['coordinator_id']);
            header('Location: coordinators.php?success=deleted');
            exit;
    }
}

$coordinators = $coordinatorRepo->getAllCoordinators();
$events = $eventRepo->getAllEvents();
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة المنسقين - نظام الباركنج</title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/admin-dashboard.css">
</head>
<body class="sidebar-fixed">
    <?php 
    $currentPage = 'coordinators';
    include __DIR__ . '/includes/sidebar.php'; 
    ?>

    <!-- Main Content -->
    <main class="main-content">
        <div class="topbar">
            <div>
                <button class="btn" id="sidebarToggle" aria-label="القائمة" style="display: none;">
                    <span class="material-icons">menu</span>
                </button>
                <h1 class="page-title">إدارة المنسقين</h1>
                <p class="page-subtitle">إضافة وإدارة منسقي الفعاليات</p>
            </div>
            <div class="topbar-actions">
                <button class="btn btn-primary" onclick="openModal()">
                    <span class="material-icons" style="font-size: 18px;">add</span>
                    إضافة منسق جديد
                </button>
            </div>
        </div>

        <?php if (isset($_GET['success'])): ?>
            <div class="alert alert-success" style="background: rgba(16, 185, 129, 0.1); border: 1px solid rgba(16, 185, 129, 0.3); color: var(--success); padding: 16px; border-radius: 12px; margin-bottom: 24px;">
                <span class="material-icons" style="vertical-align: middle; margin-left: 8px;">check_circle</span>
                <?php
                $messages = [
                    'created' => 'تم إضافة المنسق بنجاح',
                    'updated' => 'تم تحديث بيانات المنسق بنجاح',
                    'toggled' => 'تم تحديث حالة المنسق',
                    'deleted' => 'تم حذف المنسق بنجاح'
                ];
                echo $messages[$_GET['success']] ?? 'تم العملية بنجاح';
                ?>
            </div>
        <?php endif; ?>

        <!-- Coordinators Grid -->
        <div class="clients-grid" style="grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));">
            <?php if (empty($coordinators)): ?>
                <div style="grid-column: 1 / -1; text-align: center; padding: 60px; color: var(--gray);">
                    <span class="material-icons" style="font-size: 64px; opacity: 0.3;">groups</span>
                    <p style="margin-top: 16px; font-size: 18px;">لا يوجد منسقين مسجلين</p>
                    <p style="margin-top: 8px; font-size: 14px;">ابدأ بإضافة منسق جديد</p>
                </div>
            <?php else: ?>
                <?php foreach ($coordinators as $coordinator): ?>
                    <div class="client-card">
                        <div style="display: flex; align-items: start; gap: 16px; margin-bottom: 16px;">
                            <div style="width: 64px; height: 64px; border-radius: 50%; background: linear-gradient(135deg, var(--primary), var(--success)); display: flex; align-items: center; justify-content: center; color: white; font-weight: 700; font-size: 24px; flex-shrink: 0;">
                                <?php echo mb_substr($coordinator['full_name'], 0, 1); ?>
                            </div>
                            <div style="flex: 1; min-width: 0;">
                                <h3 style="font-size: 18px; font-weight: 700; color: var(--light); margin-bottom: 4px;">
                                    <?php echo htmlspecialchars($coordinator['full_name']); ?>
                                </h3>
                                <div style="display: flex; align-items: center; gap: 6px; font-size: 12px; color: var(--gray); margin-bottom: 6px;">
                                    <span class="material-icons" style="font-size: 14px;">account_circle</span>
                                    <?php echo htmlspecialchars($coordinator['username']); ?>
                                </div>
                                <span class="badge <?php echo $coordinator['is_active'] ? 'success' : 'danger'; ?>" style="font-size: 11px;">
                                    <?php echo $coordinator['is_active'] ? 'نشط' : 'معطل'; ?>
                                </span>
                            </div>
                        </div>

                        <!-- Event Info -->
                        <div style="background: rgba(99, 102, 241, 0.05); border-radius: 8px; padding: 12px; margin-bottom: 16px;">
                            <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--primary);">event</span>
                                <span style="font-size: 14px; font-weight: 600; color: var(--primary);">
                                    <?php echo htmlspecialchars($coordinator['event_name']); ?>
                                </span>
                            </div>
                            <div style="font-size: 11px; color: var(--gray); margin-right: 22px;">
                                <?php echo htmlspecialchars($coordinator['client_name']); ?>
                            </div>
                        </div>

                        <!-- Contact Info -->
                        <div class="client-details">
                            <?php if ($coordinator['phone']): ?>
                                <div class="client-detail">
                                    <span class="material-icons" style="font-size: 16px;">phone</span>
                                    <?php echo htmlspecialchars($coordinator['phone']); ?>
                                </div>
                            <?php endif; ?>
                            <?php if ($coordinator['email']): ?>
                                <div class="client-detail">
                                    <span class="material-icons" style="font-size: 16px;">email</span>
                                    <?php echo htmlspecialchars($coordinator['email']); ?>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Actions -->
                        <div class="client-actions" style="margin-top: 16px;">
                            <form method="POST" style="flex: 1;">
                                <input type="hidden" name="action" value="toggle">
                                <input type="hidden" name="coordinator_id" value="<?php echo $coordinator['id']; ?>">
                                <button type="submit" class="btn" style="width: 100%; background: <?php echo $coordinator['is_active'] ? 'rgba(239, 68, 68, 0.1)' : 'rgba(16, 185, 129, 0.1)'; ?>; color: <?php echo $coordinator['is_active'] ? '#ef4444' : 'var(--success)'; ?>;">
                                    <span class="material-icons" style="font-size: 16px;"><?php echo $coordinator['is_active'] ? 'block' : 'check_circle'; ?></span>
                                    <?php echo $coordinator['is_active'] ? 'تعطيل' : 'تفعيل'; ?>
                                </button>
                            </form>
                            <button class="btn" onclick="editCoordinator(<?php echo htmlspecialchars(json_encode($coordinator)); ?>)">
                                <span class="material-icons" style="font-size: 16px;">edit</span>
                            </button>
                            <form method="POST" onsubmit="return confirm('هل أنت متأكد من حذف هذا المنسق؟')">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="coordinator_id" value="<?php echo $coordinator['id']; ?>">
                                <button type="submit" class="btn" style="color: var(--danger);">
                                    <span class="material-icons" style="font-size: 16px;">delete</span>
                                </button>
                            </form>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </main>

    <!-- Modal -->
    <div class="modal" id="coordinatorModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">إضافة منسق جديد</h2>
                <button class="btn" onclick="closeModal()" aria-label="إغلاق">
                    <span class="material-icons">close</span>
                </button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="coordinator_id" id="coordinatorId">
                
                <div class="form-group">
                    <label>الاسم الكامل *</label>
                    <input type="text" name="full_name" id="fullName" required>
                </div>

                <div class="form-group">
                    <label>الإيفنت *</label>
                    <select name="event_id" id="eventId" required>
                        <option value="">-- اختر الإيفنت --</option>
                        <?php foreach ($events as $event): ?>
                            <option value="<?php echo $event['id']; ?>">
                                <?php echo htmlspecialchars($event['name'] . ' - ' . $event['client_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>اسم المستخدم *</label>
                        <input type="text" name="username" id="username" required>
                    </div>
                    <div class="form-group">
                        <label id="passwordLabel">كلمة المرور *</label>
                        <input type="password" name="password" id="password">
                        <small style="color: var(--gray); font-size: 11px;">اتركه فارغاً للإبقاء على كلمة المرور الحالية (عند التعديل)</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>رقم الهاتف</label>
                        <input type="text" name="phone" id="phone">
                    </div>
                    <div class="form-group">
                        <label>البريد الإلكتروني</label>
                        <input type="email" name="email" id="email">
                    </div>
                </div>
                
                <div style="display: flex; gap: 12px; margin-top: 24px;">
                    <button type="submit" class="btn btn-primary" style="flex: 1;">حفظ</button>
                    <button type="button" class="btn" onclick="closeModal()">إلغاء</button>
                </div>
            </form>
        </div>
    </div>

    <div class="sidebar-backdrop" id="sidebarBackdrop"></div>
    
    <script>
        function openModal() {
            document.getElementById('coordinatorModal').classList.add('show');
            document.getElementById('modalTitle').textContent = 'إضافة منسق جديد';
            document.getElementById('formAction').value = 'create';
            document.getElementById('coordinatorId').value = '';
            document.getElementById('fullName').value = '';
            document.getElementById('eventId').value = '';
            document.getElementById('username').value = '';
            document.getElementById('password').value = '';
            document.getElementById('password').required = true;
            document.getElementById('passwordLabel').textContent = 'كلمة المرور *';
            document.getElementById('phone').value = '';
            document.getElementById('email').value = '';
        }

        function editCoordinator(coordinator) {
            document.getElementById('coordinatorModal').classList.add('show');
            document.getElementById('modalTitle').textContent = 'تعديل بيانات المنسق';
            document.getElementById('formAction').value = 'update';
            document.getElementById('coordinatorId').value = coordinator.id;
            document.getElementById('fullName').value = coordinator.full_name;
            document.getElementById('eventId').value = coordinator.event_id;
            document.getElementById('username').value = coordinator.username;
            document.getElementById('password').value = '';
            document.getElementById('password').required = false;
            document.getElementById('passwordLabel').textContent = 'كلمة المرور (اختياري)';
            document.getElementById('phone').value = coordinator.phone || '';
            document.getElementById('email').value = coordinator.email || '';
        }

        function closeModal() {
            document.getElementById('coordinatorModal').classList.remove('show');
        }

    </script>
    <script src="assets/sidebar.js"></script>
</body>
</html>
