<?php
require_once __DIR__ . '/session_init.php';

use Parking\Auth;
use Parking\Database;
use Parking\EventRepository;
use Parking\ClientRepository;
use Parking\Installer;

$container = require __DIR__ . '/../bootstrap.php';
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));
if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

$auth = new Auth($database);
$auth->requireAuth();

$isAdmin = true;
$userFullName = $auth->getAdminFullName();

$eventRepo = new EventRepository($database);
$clientRepo = new ClientRepository($database);
$activeEvent = $eventRepo->getActiveEvent();
$clients = $clientRepo->getAllClients();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create':
            $logoPath = null;
            if (!empty($_FILES['logo']['name'])) {
                $targetDir = __DIR__ . '/assets/events/';
                if (!is_dir($targetDir)) {
                    mkdir($targetDir, 0755, true);
                }
                $extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                $filename = uniqid('event_') . '.' . $extension;
                $targetFile = $targetDir . $filename;
                if (move_uploaded_file($_FILES['logo']['tmp_name'], $targetFile)) {
                    $logoPath = 'assets/events/' . $filename;
                }
            }
            
            $eventRepo->createEvent([
                'client_id' => $_POST['client_id'],
                'name' => $_POST['name'],
                'logo' => $logoPath,
                'event_date' => $_POST['event_date'],
                'event_location' => $_POST['event_location'],
                'description' => $_POST['description'] ?? null
            ]);
            header('Location: events.php?success=created');
            exit;
            
        case 'update':
            $eventId = $_POST['event_id'];
            $event = $eventRepo->getEventById($eventId);
            $logoPath = $event['logo'];
            
            if (!empty($_FILES['logo']['name'])) {
                $targetDir = __DIR__ . '/assets/events/';
                if (!is_dir($targetDir)) {
                    mkdir($targetDir, 0755, true);
                }
                $extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                $filename = uniqid('event_') . '.' . $extension;
                $targetFile = $targetDir . $filename;
                if (move_uploaded_file($_FILES['logo']['tmp_name'], $targetFile)) {
                    // Delete old logo
                    if ($logoPath && file_exists(__DIR__ . '/' . $logoPath)) {
                        unlink(__DIR__ . '/' . $logoPath);
                    }
                    $logoPath = 'assets/events/' . $filename;
                }
            }
            
            $eventRepo->updateEvent($eventId, [
                'name' => $_POST['name'],
                'logo' => $logoPath,
                'event_date' => $_POST['event_date'],
                'event_location' => $_POST['event_location'],
                'description' => $_POST['description'] ?? null
            ]);
            header('Location: events.php?success=updated');
            exit;
            
        case 'activate':
            $eventRepo->activateEvent($_POST['event_id']);
            header('Location: events.php?success=activated');
            exit;
            
        case 'deactivate':
            $eventRepo->deactivateEvent($_POST['event_id']);
            header('Location: events.php?success=deactivated');
            exit;
            
        case 'delete':
            $event = $eventRepo->getEventById($_POST['event_id']);
            if ($event['logo'] && file_exists(__DIR__ . '/' . $event['logo'])) {
                unlink(__DIR__ . '/' . $event['logo']);
            }
            $eventRepo->deleteEvent($_POST['event_id']);
            header('Location: events.php?success=deleted');
            exit;
    }
}

$events = $eventRepo->getAllEvents();
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الفعاليات - نظام الباركنج</title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/admin-dashboard.css">
</head>
<body class="sidebar-fixed">
    <?php 
    $currentPage = 'events';
    include __DIR__ . '/includes/sidebar.php'; 
    ?>

    <!-- Main Content -->
    <main class="main-content">
        <div class="topbar">
            <div>
                <button class="btn" id="sidebarToggle" aria-label="القائمة" style="display:none;">
                    <span class="material-icons">menu</span>
                </button>
                <h1 class="page-title">إدارة الإيفنتات</h1>
                <p class="page-subtitle">إدارة فعاليات الشركات والإيفنتات</p>
            </div>
            <div class="topbar-actions">
                <button class="btn btn-primary" onclick="openCreateModal()">
                    <span class="material-icons" style="font-size: 18px;">add</span>
                    إضافة إيفنت جديد
                </button>
            </div>
        </div>

        <?php if (isset($_GET['success'])): ?>
            <div style="background: rgba(16, 185, 129, 0.1); border: 1px solid rgba(16, 185, 129, 0.3); border-radius: 8px; padding: 12px 16px; margin-bottom: 20px; color: var(--success);">
                <span class="material-icons" style="font-size: 20px; vertical-align: middle;">check_circle</span>
                تم تنفيذ العملية بنجاح
            </div>
        <?php endif; ?>

        <!-- Events Grid -->
        <div class="stats-grid" style="grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));">
            <?php foreach ($events as $event): ?>
                <div class="card" style="position: relative;">
                    <?php if ($event['is_active']): ?>
                        <div style="position: absolute; top: 12px; left: 12px; background: var(--success); color: white; padding: 4px 12px; border-radius: 20px; font-size: 11px; font-weight: 600; display: flex; align-items: center; gap: 4px;">
                            <span class="material-icons" style="font-size: 14px;">check_circle</span>
                            نشط الآن
                        </div>
                    <?php endif; ?>
                    
                    <div style="padding: 20px;">
                        <?php if ($event['logo']): ?>
                            <div style="text-align: center; margin-bottom: 16px;">
                                <img src="<?php echo htmlspecialchars($event['logo']); ?>" alt="Event Logo" style="max-width: 150px; max-height: 80px; object-fit: contain;" />
                            </div>
                        <?php endif; ?>
                        
                        <h3 style="font-size: 18px; font-weight: 700; color: var(--light); margin-bottom: 8px;">
                            <?php echo htmlspecialchars($event['name']); ?>
                        </h3>
                        
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px; padding: 8px; background: rgba(108, 99, 255, 0.05); border-radius: 6px;">
                            <span class="material-icons" style="font-size: 18px; color: var(--primary);">business</span>
                            <span style="font-size: 13px; color: var(--gray);"><?php echo htmlspecialchars($event['client_name']); ?></span>
                        </div>
                        
                        <?php if ($event['event_date']): ?>
                            <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 8px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--primary);">calendar_today</span>
                                <span style="font-size: 13px; color: var(--gray);"><?php echo htmlspecialchars($event['event_date']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($event['event_location']): ?>
                            <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 8px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--success);">location_on</span>
                                <span style="font-size: 13px; color: var(--gray);"><?php echo htmlspecialchars($event['event_location']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($event['description']): ?>
                            <p style="font-size: 12px; color: var(--gray); margin-top: 12px; padding-top: 12px; border-top: 1px solid rgba(255,255,255,0.05);">
                                <?php echo htmlspecialchars($event['description']); ?>
                            </p>
                        <?php endif; ?>
                        
                        <div style="display: flex; gap: 8px; margin-top: 16px;">
                            <?php if (!$event['is_active']): ?>
                                <form method="POST" style="flex: 1; margin: 0;">
                                    <input type="hidden" name="action" value="activate">
                                    <input type="hidden" name="event_id" value="<?php echo $event['id']; ?>">
                                    <button type="submit" class="btn btn-primary" style="width: 100%;">
                                        <span class="material-icons" style="font-size: 16px;">play_arrow</span>
                                        تفعيل
                                    </button>
                                </form>
                            <?php else: ?>
                                <form method="POST" style="flex: 1; margin: 0;">
                                    <input type="hidden" name="action" value="deactivate">
                                    <input type="hidden" name="event_id" value="<?php echo $event['id']; ?>">
                                    <button type="submit" class="btn" style="width: 100%; background: rgba(239, 68, 68, 0.1); color: #ef4444;">
                                        <span class="material-icons" style="font-size: 16px;">pause</span>
                                        إيقاف
                                    </button>
                                </form>
                            <?php endif; ?>
                            
                            <button class="btn" onclick="editEvent(<?php echo htmlspecialchars(json_encode($event), ENT_QUOTES, 'UTF-8'); ?>)">
                                <span class="material-icons" style="font-size: 16px;">edit</span>
                            </button>
                            
                            <form method="POST" style="margin: 0;" onsubmit="return confirm('هل أنت متأكد من حذف هذا الإيفنت؟');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="event_id" value="<?php echo $event['id']; ?>">
                                <button type="submit" class="btn" style="background: rgba(239, 68, 68, 0.1); color: #ef4444;">
                                    <span class="material-icons" style="font-size: 16px;">delete</span>
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
            
            <?php if (empty($events)): ?>
                <div style="grid-column: 1 / -1; text-align: center; padding: 60px; color: var(--gray);">
                    <span class="material-icons" style="font-size: 64px; opacity: 0.3;">event</span>
                    <p style="margin-top: 16px; font-size: 16px;">لا توجد إيفنتات مسجلة</p>
                    <button class="btn btn-primary" onclick="document.getElementById('createModal').style.display='flex'" style="margin-top: 16px;">
                        <span class="material-icons" style="font-size: 18px;">add</span>
                        إضافة إيفنت جديد
                    </button>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- Create Modal -->
    <div id="createModal" class="modal">
        <div class="modal-content" style="max-width: 600px;">
            <div class="modal-header">
                <h2>إضافة إيفنت جديد</h2>
                <button class="modal-close" onclick="closeCreateModal()">
                    <span class="material-icons">close</span>
                </button>
            </div>
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="create">
                
                <div class="form-group">
                    <label>الشركة</label>
                    <select name="client_id" class="input" required>
                        <option value="">اختر الشركة</option>
                        <?php foreach ($clients as $client): ?>
                            <?php if ($client['id'] != 1): // Exclude Gulf Car ?>
                                <option value="<?php echo $client['id']; ?>"><?php echo htmlspecialchars($client['name']); ?></option>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>اسم الإيفنت</label>
                    <input type="text" name="name" class="input" required>
                </div>
                
                <div class="form-group">
                    <label>شعار الإيفنت</label>
                    <input type="file" name="logo" class="input" accept="image/*">
                </div>
                
                <div class="form-group">
                    <label>تاريخ الإيفنت</label>
                    <input type="date" name="event_date" class="input">
                </div>
                
                <div class="form-group">
                    <label>موقع الإيفنت</label>
                    <input type="text" name="event_location" class="input">
                </div>
                
                <div class="form-group">
                    <label>وصف الإيفنت</label>
                    <textarea name="description" class="input" rows="3"></textarea>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn" onclick="document.getElementById('createModal').style.display='none'">إلغاء</button>
                    <button type="submit" class="btn btn-primary">
                        <span class="material-icons" style="font-size: 18px;">save</span>
                        حفظ
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Edit Modal -->
    <div id="editModal" class="modal">
        <div class="modal-content" style="max-width: 600px;">
            <div class="modal-header">
                <h2>تعديل الإيفنت</h2>
                <button class="modal-close" onclick="closeEditModal()">
                    <span class="material-icons">close</span>
                </button>
            </div>
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="event_id" id="edit_event_id">
                
                <div class="form-group">
                    <label>اسم الإيفنت</label>
                    <input type="text" name="name" id="edit_name" class="input" required>
                </div>
                
                <div class="form-group">
                    <label>شعار الإيفنت (اختياري - لتغيير الشعار)</label>
                    <input type="file" name="logo" class="input" accept="image/*">
                </div>
                
                <div class="form-group">
                    <label>تاريخ الإيفنت</label>
                    <input type="date" name="event_date" id="edit_event_date" class="input">
                </div>
                
                <div class="form-group">
                    <label>موقع الإيفنت</label>
                    <input type="text" name="event_location" id="edit_event_location" class="input">
                </div>
                
                <div class="form-group">
                    <label>وصف الإيفنت</label>
                    <textarea name="description" id="edit_description" class="input" rows="3"></textarea>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn" onclick="closeEditModal()">إلغاء</button>
                    <button type="submit" class="btn btn-primary">
                        <span class="material-icons" style="font-size: 18px;">save</span>
                        حفظ التعديلات
                    </button>
                </div>
            </form>
        </div>
    </div>

    <div class="sidebar-backdrop" id="sidebarBackdrop"></div>
    
    <script>
        function openCreateModal() {
            document.getElementById('createModal').classList.add('show');
        }

        function closeCreateModal() {
            document.getElementById('createModal').classList.remove('show');
        }

        function openEditModal() {
            document.getElementById('editModal').classList.add('show');
        }

        function closeEditModal() {
            document.getElementById('editModal').classList.remove('show');
        }

        function editEvent(event) {
            document.getElementById('edit_event_id').value = event.id;
            document.getElementById('edit_name').value = event.name;
            document.getElementById('edit_event_date').value = event.event_date || '';
            document.getElementById('edit_event_location').value = event.event_location || '';
            document.getElementById('edit_description').value = event.description || '';
            openEditModal();
        }


        
        // Close modals on outside click
        window.onclick = function(event) {
            if (event.target.classList.contains('modal')) {
                event.target.classList.remove('show');
            }
        }
    </script>
    <script src="assets/sidebar.js"></script>
</body>
</html>
