<?php
/**
 * Gulf Car Parking — Installation & Update Wizard
 *
 * Modes:
 *   fresh_install   → No .env or no DB → show config form → write .env + run migrations
 *   needs_migration → .env exists, DB connected, no tables → run migrations only
 *   installed       → Everything OK → show status + Update button
 *
 * Works on both localhost (WAMP/XAMPP) and shared hosting without changes.
 */

require_once __DIR__ . '/session_init.php';

use Parking\Config;
use Parking\Database;
use Parking\Installer;

/* ─── Bootstrap (safe — never throws) ──────────────────────────── */
$container = require __DIR__ . '/../bootstrap.php';
$config   = $container['config'];
$database = $container['database'];

$installer    = new Installer(dirname(__DIR__));
$requirements = $installer->checkRequirements();
$allMet       = array_reduce($requirements, static fn($c, $r) => $c && $r['pass'], true);

/* Detect system status safely */
$status = $installer->getStatus($database);

$messages = [];
$errors   = [];

/* ─── Handle POST Actions ──────────────────────────────────────── */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $allMet) {
    $action = $_POST['action'] ?? '';

    /* ── Fresh Install ──────────────────────────────────────────── */
    if ($action === 'install') {
        $appUrl = trim($_POST['app_url'] ?? Installer::detectAppUrl());
        $dbHost = trim($_POST['db_host'] ?? 'localhost');
        $dbPort = trim($_POST['db_port'] ?? '3306');
        $dbName = trim($_POST['db_name'] ?? '');
        $dbUser = trim($_POST['db_user'] ?? '');
        $dbPass = $_POST['db_password'] ?? '';
        $dbCharset = trim($_POST['db_charset'] ?? 'utf8mb4');

        /* Validate */
        if ($dbName === '') { $errors[] = 'اسم قاعدة البيانات مطلوب'; }
        if ($dbUser === '') { $errors[] = 'اسم مستخدم قاعدة البيانات مطلوب'; }

        if (empty($errors)) {
            /* Test connection BEFORE writing .env */
            $test = Installer::testDbConnection([
                'host'     => $dbHost,
                'port'     => $dbPort,
                'name'     => $dbName,
                'user'     => $dbUser,
                'password' => $dbPass,
                'charset'  => $dbCharset,
            ]);

            if (!$test['success']) {
                $errors[] = $test['message'];
            } else {
                $messages[] = $test['message'];

                /* Build .env values */
                $envValues = [
                    'APP_ENV'       => 'production',
                    'APP_URL'       => $appUrl,
                    'DB_DRIVER'     => 'mysql',
                    'DB_HOST'       => $dbHost,
                    'DB_PORT'       => $dbPort,
                    'DB_NAME'       => $dbName,
                    'DB_USER'       => $dbUser,
                    'DB_PASSWORD'   => $dbPass,
                    'DB_CHARSET'    => $dbCharset,
                    'MIGRATE_TOKEN' => bin2hex(random_bytes(32)),
                    'TWILIO_ACCOUNT_SID'          => '',
                    'TWILIO_AUTH_TOKEN'            => '',
                    'TWILIO_MESSAGING_SERVICE_SID' => '',
                    'TWILIO_WHATSAPP_ENABLED'     => 'false',
                    'ATTENDANT_PHONE'             => '',
                ];

                try {
                    $installer->writeEnv($envValues);
                    $messages[] = 'تم حفظ ملف الإعدادات (.env) بنجاح';

                    /* Create fresh Config+Database from new .env (includes .env.local overrides) */
                    $freshConfig   = new Config();
                    $freshDatabase = new Database($freshConfig);

                    $result   = $installer->runMigrations($freshDatabase);
                    $messages = array_merge($messages, $result['messages']);
                    $errors   = array_merge($errors, $result['errors']);

                    if (empty($result['errors'])) {
                        $status = $installer->getStatus($freshDatabase);
                    }
                } catch (\Throwable $e) {
                    $errors[] = 'خطأ في التثبيت: ' . $e->getMessage();
                }
            }
        }
    }

    /* ── Update / Repair ────────────────────────────────────────── */
    if ($action === 'update') {
        try {
            $result   = $installer->runMigrations($database);
            $messages = array_merge($messages, $result['messages']);
            $errors   = array_merge($errors, $result['errors']);

            if (empty($result['errors'])) {
                $status = $installer->getStatus($database);
            }
        } catch (\Throwable $e) {
            $errors[] = 'خطأ في التحديث: ' . $e->getMessage();
        }
    }
}

/* ─── Prepare form defaults ────────────────────────────────────── */
$currentEnv  = $installer->loadCurrentEnv();
$detectedUrl = Installer::detectAppUrl();

$formAppUrl    = $currentEnv['APP_URL'] ?: $detectedUrl;
$formDbHost    = $currentEnv['DB_HOST'] ?: 'localhost';
$formDbPort    = $currentEnv['DB_PORT'] ?: '3306';
$formDbName    = $currentEnv['DB_NAME'] ?: 'parking';
$formDbUser    = $currentEnv['DB_USER'] ?: 'root';
$formDbPass    = $currentEnv['DB_PASSWORD'] ?? '';
$formDbCharset = $currentEnv['DB_CHARSET'] ?: 'utf8mb4';

/* Helper */
function esc(string $v): string { return htmlspecialchars($v, ENT_QUOTES, 'UTF-8'); }
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>معالج التثبيت — نظام إدارة المواقف</title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <link rel="stylesheet" href="assets/theme.css">
    <style>
        *{margin:0;padding:0;box-sizing:border-box}
        body{font-family:'Segoe UI',Tahoma,sans-serif;background:linear-gradient(135deg,#0f172a,#1e293b);min-height:100vh;display:flex;align-items:center;justify-content:center;padding:20px;color:#e2e8f0}
        .wizard{max-width:700px;width:100%;background:#1e293b;border-radius:16px;border:1px solid rgba(148,163,184,.15);box-shadow:0 20px 60px rgba(0,0,0,.4);overflow:hidden}
        .wizard-header{background:linear-gradient(135deg,#6366f1,#8b5cf6);padding:28px 32px;text-align:center}
        .wizard-header h1{font-size:1.5rem;font-weight:700;color:#fff;margin:0 0 4px}
        .wizard-header p{color:rgba(255,255,255,.8);font-size:.9rem;margin:0}
        .wizard-body{padding:28px 32px}

        /* Status */
        .status-grid{display:grid;grid-template-columns:1fr 1fr;gap:10px;margin-bottom:24px}
        .status-item{display:flex;align-items:center;gap:8px;padding:10px 14px;border-radius:10px;background:rgba(30,41,59,.6);border:1px solid rgba(148,163,184,.1);font-size:.88rem}
        .status-item .material-icons{font-size:18px}
        .status-item.ok .material-icons{color:#10b981}
        .status-item.fail .material-icons{color:#ef4444}
        .status-item.warn .material-icons{color:#f59e0b}

        /* Requirements */
        .req-list{list-style:none;padding:0;margin:0 0 24px}
        .req-list li{display:flex;align-items:center;gap:8px;padding:8px 12px;border-bottom:1px solid rgba(148,163,184,.08);font-size:.88rem}
        .req-list li:last-child{border-bottom:none}
        .req-list .material-icons{font-size:18px}
        .req-list .pass .material-icons{color:#10b981}
        .req-list .fail .material-icons{color:#ef4444}
        .req-list .hint{color:#94a3b8;font-size:.8rem;margin-right:auto;text-align:left;direction:ltr}

        /* Section */
        .section{margin-bottom:24px}
        .section-title{font-size:1rem;font-weight:600;margin-bottom:14px;display:flex;align-items:center;gap:8px;color:#e2e8f0}
        .section-title .material-icons{font-size:20px;color:#6366f1}

        /* Alerts */
        .alert{padding:14px 18px;border-radius:10px;margin-bottom:16px;font-size:.88rem;display:flex;align-items:flex-start;gap:10px;line-height:1.6}
        .alert .material-icons{font-size:20px;flex-shrink:0;margin-top:2px}
        .alert-success{background:rgba(16,185,129,.12);border:1px solid rgba(16,185,129,.3);color:#6ee7b7}
        .alert-success .material-icons{color:#10b981}
        .alert-error{background:rgba(239,68,68,.12);border:1px solid rgba(239,68,68,.3);color:#fca5a5}
        .alert-error .material-icons{color:#ef4444}
        .alert-info{background:rgba(99,102,241,.12);border:1px solid rgba(99,102,241,.3);color:#a5b4fc}
        .alert-info .material-icons{color:#6366f1}

        /* Form */
        .form-group{margin-bottom:16px}
        .form-group label{display:block;font-size:.85rem;font-weight:500;margin-bottom:6px;color:#cbd5e1}
        .form-group input,.form-group select{width:100%;padding:10px 14px;border-radius:8px;border:1px solid rgba(148,163,184,.2);background:#0f172a;color:#e2e8f0;font-size:.9rem;transition:border .2s}
        .form-group input:focus,.form-group select:focus{outline:none;border-color:#6366f1;box-shadow:0 0 0 3px rgba(99,102,241,.15)}
        .form-group .hint{display:block;margin-top:4px;font-size:.78rem;color:#64748b}
        .form-row{display:grid;grid-template-columns:1fr 1fr;gap:14px}

        /* Buttons */
        .btn{display:inline-flex;align-items:center;justify-content:center;gap:8px;padding:12px 24px;border-radius:10px;font-size:.95rem;font-weight:600;border:none;cursor:pointer;transition:all .2s;text-decoration:none}
        .btn-primary{background:linear-gradient(135deg,#6366f1,#8b5cf6);color:#fff;width:100%}
        .btn-primary:hover{transform:translateY(-1px);box-shadow:0 8px 24px rgba(99,102,241,.3)}
        .btn-primary:disabled{opacity:.5;cursor:not-allowed;transform:none}
        .btn-success{background:linear-gradient(135deg,#10b981,#059669);color:#fff}
        .btn-success:hover{transform:translateY(-1px);box-shadow:0 8px 24px rgba(16,185,129,.3)}
        .btn-outline{background:transparent;border:1px solid rgba(148,163,184,.3);color:#e2e8f0}
        .btn-outline:hover{background:rgba(148,163,184,.1)}

        .actions{display:flex;gap:12px;margin-top:24px;flex-wrap:wrap}
        .actions .btn{flex:1;min-width:140px}

        /* Installed badge */
        .installed-badge{text-align:center;padding:32px 20px;margin-bottom:20px}
        .installed-badge .material-icons{font-size:64px;color:#10b981;display:block;margin:0 auto 12px}
        .installed-badge h2{color:#10b981;font-size:1.3rem;margin-bottom:6px}
        .installed-badge p{color:#94a3b8;font-size:.9rem}

        /* Responsive */
        @media(max-width:600px){
            .wizard-body{padding:20px 16px}
            .form-row{grid-template-columns:1fr}
            .status-grid{grid-template-columns:1fr}
            .actions{flex-direction:column}
        }
    </style>
</head>
<body>

<div class="wizard">
    <!-- Header -->
    <div class="wizard-header">
        <h1><span class="material-icons" style="vertical-align:middle;margin-left:6px">directions_car</span> نظام إدارة المواقف</h1>
        <p>معالج التثبيت والتحديث</p>
    </div>

    <div class="wizard-body">

        <!-- Messages -->
        <?php foreach ($messages as $msg): ?>
            <div class="alert alert-success">
                <span class="material-icons">check_circle</span>
                <span><?= esc($msg) ?></span>
            </div>
        <?php endforeach; ?>

        <?php foreach ($errors as $err): ?>
            <div class="alert alert-error">
                <span class="material-icons">error</span>
                <span><?= esc($err) ?></span>
            </div>
        <?php endforeach; ?>

        <!-- ═══ INSTALLED ═══════════════════════════════════════ -->
        <?php if ($status['mode'] === 'installed'): ?>

            <div class="installed-badge">
                <span class="material-icons">verified</span>
                <h2>النظام مثبت ويعمل بنجاح</h2>
                <p>يمكنك البدء في استخدام النظام أو تشغيل التحديث</p>
            </div>

            <!-- Status Grid -->
            <div class="status-grid">
                <div class="status-item ok"><span class="material-icons">settings</span> ملف الإعدادات (.env)</div>
                <div class="status-item ok"><span class="material-icons">storage</span> اتصال قاعدة البيانات</div>
                <div class="status-item ok"><span class="material-icons">table_chart</span> جداول البيانات</div>
                <div class="status-item <?= $status['admin_exists'] ? 'ok' : 'warn' ?>">
                    <span class="material-icons"><?= $status['admin_exists'] ? 'admin_panel_settings' : 'warning' ?></span>
                    <?= $status['admin_exists'] ? 'حساب المدير' : 'لا يوجد حساب مدير' ?>
                </div>
            </div>

            <!-- Action buttons -->
            <div class="actions">
                <a href="login.php" class="btn btn-primary">
                    <span class="material-icons">login</span> تسجيل الدخول
                </a>
                <form method="post" style="flex:1;min-width:140px;display:flex">
                    <input type="hidden" name="action" value="update">
                    <button type="submit" class="btn btn-outline" style="width:100%">
                        <span class="material-icons">system_update</span> تحديث / إصلاح
                    </button>
                </form>
            </div>

            <?php if ($status['admin_exists']): ?>
                <div class="alert alert-info" style="margin-top:20px">
                    <span class="material-icons">info</span>
                    <div>
                        <strong>بيانات الدخول الافتراضية:</strong><br>
                        المستخدم: <code style="color:#a5b4fc">admin</code> &nbsp;—&nbsp;
                        كلمة المرور: <code style="color:#a5b4fc">admin123</code>
                    </div>
                </div>
            <?php endif; ?>

        <!-- ═══ NEEDS MIGRATION ═════════════════════════════════ -->
        <?php elseif ($status['mode'] === 'needs_migration'): ?>

            <div class="status-grid">
                <div class="status-item ok"><span class="material-icons">settings</span> ملف الإعدادات (.env)</div>
                <div class="status-item <?= $status['db_connected'] ? 'ok' : 'fail' ?>">
                    <span class="material-icons"><?= $status['db_connected'] ? 'storage' : 'error' ?></span>
                    <?= $status['db_connected'] ? 'اتصال قاعدة البيانات' : 'فشل الاتصال' ?>
                </div>
                <div class="status-item warn"><span class="material-icons">warning</span> الجداول غير موجودة</div>
                <div class="status-item fail"><span class="material-icons">error</span> يحتاج تهيئة</div>
            </div>

            <?php if ($status['error']): ?>
                <div class="alert alert-error">
                    <span class="material-icons">error</span>
                    <span><?= esc($status['error']) ?></span>
                </div>
            <?php endif; ?>

            <?php if ($status['db_connected']): ?>
                <!-- DB connected but no tables → just run migrations -->
                <div class="alert alert-info">
                    <span class="material-icons">info</span>
                    <span>الاتصال بقاعدة البيانات ناجح لكن الجداول غير موجودة. اضغط لتهيئة قاعدة البيانات.</span>
                </div>
                <form method="post">
                    <input type="hidden" name="action" value="update">
                    <button type="submit" class="btn btn-primary">
                        <span class="material-icons">build</span> تهيئة قاعدة البيانات
                    </button>
                </form>
            <?php else: ?>
                <!-- DB not connected → show full form to re-configure -->
                <div class="alert alert-error">
                    <span class="material-icons">error</span>
                    <span>لا يمكن الاتصال بقاعدة البيانات. أعد إدخال البيانات أو تحقق من ملف .env</span>
                </div>
                <?php /* Show the same install form below */ ?>
                <?php $status['mode'] = '__show_form__'; ?>
            <?php endif; ?>

        <!-- ═══ FRESH INSTALL ═══════════════════════════════════ -->
        <?php endif; ?>

        <?php if ($status['mode'] === 'fresh_install' || ($status['mode'] ?? '') === '__show_form__'): ?>

            <!-- Requirements -->
            <div class="section">
                <div class="section-title">
                    <span class="material-icons">checklist</span> متطلبات الخادم
                </div>
                <ul class="req-list">
                    <?php foreach ($requirements as $r): ?>
                        <li class="<?= $r['pass'] ? 'pass' : 'fail' ?>">
                            <span class="material-icons"><?= $r['pass'] ? 'check_circle' : 'cancel' ?></span>
                            <span><?= esc($r['label']) ?></span>
                            <span class="hint"><?= esc($r['hint']) ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <?php if (!$allMet): ?>
                    <div class="alert alert-error">
                        <span class="material-icons">block</span>
                        <span>لا يمكن المتابعة — تحقق من المتطلبات أعلاه</span>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Install Form -->
            <form method="post" autocomplete="off">
                <input type="hidden" name="action" value="install">

                <div class="section">
                    <div class="section-title">
                        <span class="material-icons">language</span> رابط التطبيق
                    </div>
                    <div class="form-group">
                        <label>APP_URL</label>
                        <input type="url" name="app_url" value="<?= esc($formAppUrl) ?>" required <?= !$allMet ? 'disabled' : '' ?>
                               placeholder="http://localhost/parking/public">
                        <span class="hint">الرابط المكتشف تلقائياً: <?= esc($detectedUrl) ?></span>
                    </div>
                </div>

                <div class="section">
                    <div class="section-title">
                        <span class="material-icons">storage</span> إعدادات قاعدة البيانات (MySQL)
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>اسم الخادم (Host)</label>
                            <input type="text" name="db_host" value="<?= esc($formDbHost) ?>" placeholder="localhost" <?= !$allMet ? 'disabled' : '' ?>>
                        </div>
                        <div class="form-group">
                            <label>المنفذ (Port)</label>
                            <input type="text" name="db_port" value="<?= esc($formDbPort) ?>" placeholder="3306" <?= !$allMet ? 'disabled' : '' ?>>
                        </div>
                    </div>

                    <div class="form-group">
                        <label>اسم قاعدة البيانات</label>
                        <input type="text" name="db_name" value="<?= esc($formDbName) ?>" placeholder="parking" required <?= !$allMet ? 'disabled' : '' ?>>
                        <span class="hint">سيتم إنشاؤها تلقائياً إذا لم تكن موجودة</span>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>اسم المستخدم</label>
                            <input type="text" name="db_user" value="<?= esc($formDbUser) ?>" placeholder="root" required <?= !$allMet ? 'disabled' : '' ?>>
                        </div>
                        <div class="form-group">
                            <label>كلمة المرور</label>
                            <input type="password" name="db_password" value="<?= esc($formDbPass) ?>" <?= !$allMet ? 'disabled' : '' ?>>
                            <span class="hint">اتركه فارغاً إذا لم يكن هناك كلمة مرور</span>
                        </div>
                    </div>

                    <div class="form-group">
                        <label>ترميز الأحرف (Charset)</label>
                        <select name="db_charset" <?= !$allMet ? 'disabled' : '' ?>>
                            <option value="utf8mb4" <?= $formDbCharset === 'utf8mb4' ? 'selected' : '' ?>>UTF-8 (utf8mb4) — موصى به</option>
                            <option value="utf8" <?= $formDbCharset === 'utf8' ? 'selected' : '' ?>>UTF-8 (utf8)</option>
                        </select>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary" <?= !$allMet ? 'disabled' : '' ?>>
                    <span class="material-icons">install_desktop</span> تثبيت النظام
                </button>

                <div class="alert alert-info" style="margin-top:20px">
                    <span class="material-icons">info</span>
                    <div>
                        <strong>ملاحظة:</strong> سيتم اختبار الاتصال بقاعدة البيانات أولاً قبل حفظ الإعدادات.
                        إذا لم تكن قاعدة البيانات موجودة سيتم إنشاؤها تلقائياً.
                        <br><br>
                        <strong>بعد التثبيت:</strong> اسم المستخدم: <code style="color:#a5b4fc">admin</code> — كلمة المرور: <code style="color:#a5b4fc">admin123</code>
                    </div>
                </div>
            </form>

        <?php endif; ?>

    </div><!-- .wizard-body -->
</div><!-- .wizard -->

</body>
</html>
