<?php

use Parking\Helpers;
use Parking\Installer;
use Parking\ParkingRepository;
use Parking\Notifier;

require_once __DIR__ . '/session_init.php';

$container = require __DIR__ . '/../bootstrap.php';
$config = $container['config'];
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));

if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}

$customerName = trim($_POST['customer_name'] ?? '');
$phoneNumber = trim($_POST['phone_number'] ?? '');
$licensePlate = trim($_POST['license_plate'] ?? '');
$valetName = trim($_POST['valet_name'] ?? '');
$eventId = !empty($_POST['event_id']) ? (int)$_POST['event_id'] : null;

// Get coordinator_id if coordinator is logged in
$coordinatorId = isset($_SESSION['coordinator_id']) ? (int)$_SESSION['coordinator_id'] : null;

if ($customerName === '' || $phoneNumber === '' || $licensePlate === '' || $valetName === '') {
    header('Location: index.php?error=missing');
    exit;
}

// Handle photo uploads
$uploadedPhotos = [];
if (!empty($_FILES['photos']['name'][0])) {
    $uploadDir = __DIR__ . '/assets/vehicles/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'];
    $maxFileSize = 5 * 1024 * 1024; // 5MB
    
    foreach ($_FILES['photos']['tmp_name'] as $key => $tmpName) {
        if ($_FILES['photos']['error'][$key] === UPLOAD_ERR_OK) {
            $fileType = $_FILES['photos']['type'][$key];
            $fileSize = $_FILES['photos']['size'][$key];
            
            // Validate file type and size
            if (in_array($fileType, $allowedTypes) && $fileSize <= $maxFileSize) {
                $ext = pathinfo($_FILES['photos']['name'][$key], PATHINFO_EXTENSION);
                $filename = 'vehicle_' . time() . '_' . uniqid() . '.' . $ext;
                $uploadPath = $uploadDir . $filename;
                
                if (move_uploaded_file($tmpName, $uploadPath)) {
                    $uploadedPhotos[] = 'assets/vehicles/' . $filename;
                }
            }
        }
    }
}

$repository = new ParkingRepository($database);
$ticketId = $repository->registerVehicle($customerName, $phoneNumber, $licensePlate, $valetName, $eventId, $coordinatorId, $uploadedPhotos);
$requestToken = Helpers::randomToken();

$baseUrl = Helpers::baseUrl($config);
$requestUrl = $baseUrl . '/request.php?token=' . $requestToken;
$repository->storeRequestToken($ticketId, $requestToken, $requestUrl);

$sent = false;
$error = null;

$hasTwilioConfig = $config->get('TWILIO_ACCOUNT_SID') && $config->get('TWILIO_AUTH_TOKEN') && $config->get('TWILIO_MESSAGING_SERVICE_SID');

if ($hasTwilioConfig) {
    try {
        $notifier = new Notifier($config);
        $body = "مرحباً {$customerName}! لطلب سيارتك اضغط على الرابط: {$requestUrl}";
        $notifier->sendMessage($phoneNumber, $body);

        $attendantPhone = $config->get('ATTENDANT_PHONE');
        if ($attendantPhone) {
            $notifier->sendMessage($attendantPhone, "تنبيه: تم تسجيل سيارة جديدة للعميل {$customerName} (لوحة {$licensePlate}).");
        }

        $sent = true;
    } catch (Throwable $exception) {
        $error = $exception->getMessage();
    }
}

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>تم تسجيل السيارة</title>
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/styles.css">
</head>
<body class="rtl">
    <main class="layout">
        <section class="card">
            <h1>تم تسجيل السيارة بنجاح</h1>
            <p>رقم التذكرة: <strong>#<?php echo htmlspecialchars((string) $ticketId); ?></strong></p>
            <p>الرابط الخاص بالعميل:</p>
            <p><a href="<?php echo htmlspecialchars($requestUrl); ?>"><?php echo htmlspecialchars($requestUrl); ?></a></p>
            <?php if ($sent): ?>
                <p class="tip">تم إرسال الرسالة للعميل.</p>
            <?php elseif ($error): ?>
                <p class="tip">تعذر إرسال الرسالة: <?php echo htmlspecialchars($error); ?></p>
            <?php else: ?>
                <p class="tip">لم يتم إرسال الرسالة تلقائياً. تحقق من إعدادات Twilio.</p>
            <?php endif; ?>
            <div style="margin-top: 1.5rem; display: flex; gap: 1rem;">
                <a href="index.php"><button type="button">تسجيل سيارة جديدة</button></a>
                <a href="dashboard.php"><button type="button">فتح لوحة المتابعة</button></a>
            </div>
        </section>
    </main>
</body>
</html>
