<?php

declare(strict_types=1);

use Parking\Installer;
use Parking\Notifier;
use Parking\ParkingRepository;
use Parking\EventRepository;

$container = require __DIR__ . '/../bootstrap.php';
$config = $container['config'];
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));

if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

$repository = new ParkingRepository($database);
$eventRepo = new EventRepository($database);
$token = trim($_GET['token'] ?? '');

if ($token === '') {
    http_response_code(400);
    echo 'رمز غير صالح.';
    exit;
}

$ticket = $repository->findByToken($token);

if (!$ticket) {
    http_response_code(404);
    echo 'لم يتم العثور على معلومات السيارة. الرجاء التواصل مع المنسق.';
    exit;
}

$statusMessage = null;
$errorMessage = null;
$justRequested = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($ticket['status'] === 'returned') {
        $errorMessage = 'تم تسليم السيارة بالفعل.';
    } else {
        $alreadyRequested = $ticket['status'] === 'requested';

        if (!$alreadyRequested) {
            try {
                $repository->markRequested((int) $ticket['id']);
                $justRequested = true;
            } catch (Throwable $exception) {
                $errorMessage = 'تعذر تنفيذ الطلب: ' . $exception->getMessage();
            }
        }

        $ticket = $repository->findByToken($token) ?? $ticket;

        if (!$errorMessage) {
            $statusMessage = $alreadyRequested
                ? 'تم استلام طلبك مسبقاً وسيتم تجهيز السيارة قريباً.'
                : 'تم إرسال الطلب للمنسق، سيتم تجهيز سيارتك فوراً.';
        }
    }

    if ($justRequested && !$errorMessage) {
        $hasTwilioConfig = $config->get('TWILIO_ACCOUNT_SID') && $config->get('TWILIO_AUTH_TOKEN') && $config->get('TWILIO_MESSAGING_SERVICE_SID');

        if ($hasTwilioConfig) {
            try {
                $notifier = new Notifier($config);
                $notifier->sendMessage($ticket['phone_number'], 'تم استلام طلبك وسيتم تجهيز السيارة فوراً. شكراً لثقتك بنا.');

                $attendantPhone = $config->get('ATTENDANT_PHONE');
                if ($attendantPhone) {
                    $notifier->sendMessage($attendantPhone, 'تنبيه: العميل ' . $ticket['customer_name'] . ' طلب السيارة رقم ' . $ticket['license_plate']);
                }
            } catch (Throwable $exception) {
                $errorMessage = 'تم تنفيذ الطلب لكن تعذر إرسال التنبيهات: ' . $exception->getMessage();
            }
        }
    }
}

$isRequested = $ticket['status'] === 'requested';
$isReturned = $ticket['status'] === 'returned';

// Get event info
$eventInfo = null;
if (!empty($ticket['event_id'])) {
    $eventInfo = $eventRepo->getEventById($ticket['event_id']);
}

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=5.0, user-scalable=yes">
    <meta name="theme-color" content="#0f172a">
    <meta name="description" content="طلب استرجاع السيارة - خدمة إيقاف السيارات الفاخرة">
    <title>طلب السيارة - نظام الباركنج</title>
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <style>
        * {
            -webkit-tap-highlight-color: transparent;
        }

        body {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
            background: #0f172a;
            position: relative;
            overflow: hidden;
        }

        /* Dark Mode Background Pattern */
        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: 
                radial-gradient(circle at 20% 50%, rgba(99, 102, 241, 0.15) 0%, transparent 50%),
                radial-gradient(circle at 80% 80%, rgba(16, 185, 129, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 40% 20%, rgba(245, 158, 11, 0.08) 0%, transparent 50%);
            z-index: 0;
            pointer-events: none;
        }

        body > * {
            position: relative;
            z-index: 1;
        }

        .request-container {
            max-width: 600px;
            width: 100%;
        }

        .brand-header {
            text-align: center;
            margin-bottom: 20px;
            animation: fadeInDown 0.6s ease-out;
        }

        .brand-logo-wrapper {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 12px;
            margin-bottom: 12px;
        }

        .brand-logo {
            height: 70px;
            width: auto;
            filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.2));
        }

        .brand-title {
            font-size: 28px;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .brand-subtitle {
            font-size: 13px;
            color: #d1d5db;
            margin-top: 6px;
            font-weight: 500;
        }

        /* Merged Event Banner with Logos - Single Row */
        .event-banner-merged {
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.15), rgba(16, 185, 129, 0.1));
            border: 1px solid rgba(99, 102, 241, 0.3);
            border-radius: 16px;
            padding: 20px;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.2);
            backdrop-filter: blur(10px);
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 20px;
            direction: rtl;
        }

        .event-info-merged {
            flex: 1;
            text-align: right;
            min-width: 0;
        }

        .event-title {
            font-size: 20px;
            font-weight: 700;
            color: #f8fafc;
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            justify-content: flex-start;
            gap: 8px;
            line-height: 1.3;
        }

        .event-title .material-icons {
            color: #a5b4fc;
            font-size: 24px;
        }

        .event-meta {
            display: flex;
            flex-wrap: wrap;
            justify-content: flex-start;
            gap: 12px;
            font-size: 13px;
            color: #d1d5db;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .meta-item .material-icons {
            font-size: 16px;
            color: #a5b4fc;
        }

        .logos-container {
            display: flex;
            align-items: center;
            gap: 16px;
            flex-shrink: 0;
        }

        .main-logo {
            height: 70px;
            width: auto;
            object-fit: contain;
            filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.3));
        }

        .logo-divider {
            width: 2px;
            height: 55px;
            background: linear-gradient(180deg, transparent, rgba(99, 102, 241, 0.5), transparent);
        }

        .event-logo-merged {
            height: 70px;
            width: auto;
            max-width: 140px;
            object-fit: contain;
            filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.3));
        }

        .request-card {
            background: rgba(30, 41, 59, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 18px;
            padding: 32px;
            box-shadow: 
                0 20px 60px rgba(0, 0, 0, 0.5),
                0 0 0 1px rgba(99, 102, 241, 0.1) inset;
            border: 1px solid rgba(99, 102, 241, 0.2);
            animation: fadeInUp 0.6s ease-out;
        }

        @keyframes fadeInDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .welcome-section {
            text-align: center;
            margin-bottom: 22px;
            padding-bottom: 22px;
            border-bottom: 1px solid rgba(99, 102, 241, 0.2);
            background: linear-gradient(180deg, rgba(99, 102, 241, 0.05) 0%, transparent 100%);
            margin: -32px -32px 22px -32px;
            padding: 24px 32px 22px 32px;
            border-radius: 18px 18px 0 0;
        }

        .welcome-title {
            font-size: 23px;
            font-weight: 700;
            color: #f8fafc;
            margin-bottom: 10px;
            text-shadow: 0 2px 10px rgba(0, 0, 0, 0.3);
            line-height: 1.3;
        }

        .welcome-message {
            font-size: 15px;
            color: #d1d5db;
            line-height: 1.5;
        }

        .info-grid {
            display: grid;
            gap: 14px;
            margin-bottom: 22px;
        }

        .info-item {
            display: flex;
            align-items: center;
            gap: 14px;
            padding: 16px;
            background: rgba(15, 23, 42, 0.6);
            border: 1px solid rgba(99, 102, 241, 0.3);
            border-radius: 12px;
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
        }

        .info-item:hover {
            background: rgba(99, 102, 241, 0.15);
            border-color: rgba(99, 102, 241, 0.5);
            transform: translateX(-4px);
            box-shadow: 0 8px 20px rgba(99, 102, 241, 0.3);
        }

        .info-icon {
            font-size: 28px;
            color: #a5b4fc;
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.3), rgba(99, 102, 241, 0.15));
            padding: 10px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 12px rgba(99, 102, 241, 0.2);
        }

        .info-content {
            flex: 1;
        }

        .info-label {
            font-size: 12px;
            color: #9ca3af;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 5px;
            font-weight: 500;
        }

        .info-value {
            font-size: 17px;
            font-weight: 700;
            color: #f3f4f6;
        }

        .status-badge {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 6px 14px;
            border-radius: 18px;
            font-size: 13px;
            font-weight: 600;
        }

        .status-badge.parked {
            background: linear-gradient(135deg, rgba(59, 130, 246, 0.25), rgba(59, 130, 246, 0.15));
            color: #bfdbfe;
            border: 1px solid rgba(59, 130, 246, 0.6);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .status-badge.requested {
            background: linear-gradient(135deg, rgba(245, 158, 11, 0.25), rgba(245, 158, 11, 0.15));
            color: #fde68a;
            border: 1px solid rgba(245, 158, 11, 0.6);
            box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3);
        }

        .status-badge.returned {
            background: linear-gradient(135deg, rgba(16, 185, 129, 0.25), rgba(16, 185, 129, 0.15));
            color: #a7f3d0;
            border: 1px solid rgba(16, 185, 129, 0.6);
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
        }

        .alert {
            padding: 16px;
            border-radius: 12px;
            margin-bottom: 18px;
            display: flex;
            align-items: flex-start;
            gap: 10px;
            animation: slideIn 0.3s ease-out;
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .alert-icon {
            font-size: 24px;
            margin-top: 2px;
        }

        .alert-content {
            flex: 1;
        }

        .alert.success {
            background: linear-gradient(135deg, rgba(16, 185, 129, 0.2), rgba(16, 185, 129, 0.1));
            border: 1px solid rgba(16, 185, 129, 0.6);
            color: #a7f3d0;
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.2);
        }

        .alert.success .alert-icon {
            color: #6ee7b7;
            filter: drop-shadow(0 2px 4px rgba(16, 185, 129, 0.5));
        }

        .alert.error {
            background: linear-gradient(135deg, rgba(239, 68, 68, 0.2), rgba(239, 68, 68, 0.1));
            border: 1px solid rgba(239, 68, 68, 0.6);
            color: #fecaca;
            box-shadow: 0 4px 12px rgba(239, 68, 68, 0.2);
        }

        .alert.error .alert-icon {
            color: #fca5a5;
            filter: drop-shadow(0 2px 4px rgba(239, 68, 68, 0.5));
        }

        .action-section {
            margin-top: 22px;
        }

        .request-button {
            width: 100%;
            padding: 16px;
            font-size: 17px;
            font-weight: 600;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 12px;
            transition: all 0.3s ease;
            background: linear-gradient(135deg, #6366f1 0%, #ec4899 50%, #f59e0b 100%);
            background-size: 200% 100%;
            color: white;
            box-shadow: 
                0 4px 20px rgba(99, 102, 241, 0.4),
                0 0 0 1px rgba(255, 255, 255, 0.1) inset;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
            position: relative;
            overflow: hidden;
            -webkit-touch-callout: none;
            -webkit-user-select: none;
            user-select: none;
            touch-action: manipulation;
            min-height: 56px;
        }

        .request-button::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
            transition: left 0.5s;
        }

        .request-button:hover:not(:disabled)::before {
            left: 100%;
        }

        .request-button:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 
                0 8px 30px rgba(99, 102, 241, 0.6),
                0 0 0 1px rgba(255, 255, 255, 0.2) inset;
            background-position: 100% 0;
        }

        .request-button:active:not(:disabled) {
            transform: translateY(0) scale(0.98);
            box-shadow: 
                0 2px 12px rgba(99, 102, 241, 0.4),
                0 0 0 1px rgba(255, 255, 255, 0.15) inset;
        }

        @media (hover: none) and (pointer: coarse) {
            /* Touch devices only */
            .request-button:hover:not(:disabled) {
                transform: none;
            }
            
            .request-button:active:not(:disabled) {
                transform: scale(0.96);
                background-position: 100% 0;
            }
        }

        .request-button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            background: linear-gradient(135deg, #475569, #334155);
            box-shadow: none;
        }

        .request-button .material-icons {
            font-size: 28px;
        }

        /* Tablet styles */
        @media (max-width: 768px) {
            body {
                padding: 16px;
            }

            .request-card {
                padding: 32px 24px;
            }

            .event-banner-merged {
                padding: 16px;
                gap: 16px;
            }

            .main-logo,
            .event-logo-merged {
                height: 60px;
            }

            .logo-divider {
                height: 48px;
            }

            .event-title {
                font-size: 18px;
            }

            .event-title .material-icons {
                font-size: 22px;
            }

            .event-meta {
                font-size: 12px;
                gap: 10px;
            }

            .meta-item .material-icons {
                font-size: 15px;
            }

            .welcome-title {
                font-size: 23px;
            }

            .welcome-message {
                font-size: 15px;
            }

            .info-value {
                font-size: 17px;
            }

            .request-button {
                font-size: 17px;
                padding: 16px;
            }
        }

        /* Mobile styles */
        @media (max-width: 640px) {
            body {
                padding: 12px;
            }

            .request-container {
                max-width: 100%;
            }

            .brand-header {
                margin-bottom: 20px;
            }

            .brand-logo {
                height: 60px;
            }

            .brand-title {
                font-size: 22px;
            }

            .brand-subtitle {
                font-size: 12px;
            }

            .event-banner-merged {
                flex-direction: column;
                padding: 16px;
                gap: 12px;
                text-align: center;
            }

            .logos-container {
                gap: 12px;
            }

            .main-logo,
            .event-logo-merged {
                height: 55px;
            }

            .logo-divider {
                height: 45px;
            }

            .event-info-merged {
                text-align: center;
            }

            .event-title {
                font-size: 17px;
                justify-content: center;
            }

            .event-title .material-icons {
                font-size: 20px;
            }

            .event-meta {
                justify-content: center;
                font-size: 12px;
                gap: 8px;
            }

            .meta-item .material-icons {
                font-size: 15px;
            }

            .request-card {
                padding: 22px 18px;
                border-radius: 16px;
            }

            .welcome-section {
                margin: -22px -18px 18px -18px;
                padding: 20px 18px 18px 18px;
                border-radius: 16px 16px 0 0;
            }

            .welcome-title {
                font-size: 20px;
                margin-bottom: 8px;
            }

            .welcome-message {
                font-size: 14px;
            }

            .alert {
                padding: 16px;
                font-size: 14px;
            }

            .alert-icon {
                font-size: 20px;
            }

            .info-item {
                padding: 16px;
                gap: 12px;
            }

            .info-icon {
                font-size: 28px;
                padding: 10px;
            }

            .info-label {
                font-size: 12px;
                margin-bottom: 4px;
            }

            .info-value {
                font-size: 16px;
            }

            .status-badge {
                font-size: 13px;
                padding: 6px 12px;
            }

            .status-badge .material-icons {
                font-size: 16px !important;
            }

            .request-button {
                font-size: 16px;
                padding: 16px;
                gap: 10px;
            }

            .request-button .material-icons {
                font-size: 24px;
            }
        }

        /* Extra small mobile */
        @media (max-width: 375px) {
            body {
                padding: 8px;
            }

            .brand-logo {
                height: 55px;
            }

            .brand-title {
                font-size: 20px;
            }

            .brand-subtitle {
                font-size: 11px;
            }

            .event-banner-merged {
                flex-direction: column;
                padding: 14px;
                gap: 10px;
            }

            .logos-container {
                gap: 10px;
            }

            .main-logo,
            .event-logo-merged {
                height: 50px;
            }

            .logo-divider {
                height: 40px;
            }

            .event-title {
                font-size: 16px;
            }

            .event-title .material-icons {
                font-size: 19px;
            }

            .event-meta {
                font-size: 11px;
                gap: 6px;
            }

            .meta-item .material-icons {
                font-size: 14px;
            }

            .request-card {
                padding: 20px 16px;
            }

            .welcome-section {
                margin: -20px -16px 16px -16px;
                padding: 18px 16px 16px 16px;
            }

            .welcome-title {
                font-size: 18px;
            }

            .welcome-message {
                font-size: 13px;
            }

            .info-item {
                padding: 14px;
            }

            .info-icon {
                font-size: 26px;
                padding: 8px;
            }

            .info-label {
                font-size: 11px;
            }

            .info-value {
                font-size: 15px;
            }

            .status-badge {
                font-size: 12px;
                padding: 5px 10px;
            }

            .request-button {
                font-size: 15px;
                padding: 14px;
            }

            .request-button .material-icons {
                font-size: 22px;
            }
        }

        /* Landscape mobile */
        @media (max-width: 896px) and (orientation: landscape) {
            body {
                padding: 12px;
            }

            .brand-header {
                margin-bottom: 16px;
            }

            .event-banner-merged {
                padding: 14px 18px;
                gap: 14px;
            }

            .main-logo,
            .event-logo-merged {
                height: 55px;
            }

            .logo-divider {
                height: 45px;
            }

            .event-title {
                font-size: 17px;
            }

            .event-meta {
                font-size: 12px;
                gap: 10px;
            }

            .welcome-section {
                padding: 24px 32px 20px 32px;
                margin: -32px -32px 20px -32px;
            }

            .welcome-title {
                font-size: 22px;
            }

            .request-card {
                max-width: 600px;
                margin: 0 auto;
            }
        }
    </style>
</head>
<body class="rtl">
    <div class="request-container">
        <!-- Combined Header with Event Info -->
        <div class="brand-header">
            <?php if ($eventInfo): ?>
                <!-- Event Banner with Logos -->
                <div class="event-banner-merged">
                    <div class="logos-container">
                        <img src="assets/logo.png" alt="Gulf Car" class="main-logo" />
                        <?php if (!empty($eventInfo['logo'])): ?>
                            <div class="logo-divider"></div>
                            <img src="<?php echo htmlspecialchars($eventInfo['logo']); ?>" alt="Event Logo" class="event-logo-merged" />
                        <?php endif; ?>
                    </div>
                    
                    <div class="event-info-merged">
                        <h1 class="event-title">
                            <span class="material-icons">event</span>
                            <span><?php echo htmlspecialchars($eventInfo['name']); ?></span>
                        </h1>
                        
                        <div class="event-meta">
                            <?php if (!empty($eventInfo['event_date'])): ?>
                                <div class="meta-item">
                                    <span class="material-icons">calendar_today</span>
                                    <span><?php echo htmlspecialchars($eventInfo['event_date']); ?></span>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($eventInfo['event_location'])): ?>
                                <div class="meta-item">
                                    <span class="material-icons">location_on</span>
                                    <span><?php echo htmlspecialchars($eventInfo['event_location']); ?></span>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <!-- Default Header (no event) -->
                <div class="brand-logo-wrapper">
                    <img src="assets/logo.png" alt="Gulf Car" class="brand-logo" />
                </div>
                <h1 class="brand-title">شركة الخليج</h1>
                <p class="brand-subtitle">خدمة إيقاف السيارات الفاخرة</p>
            <?php endif; ?>
        </div>

        <!-- Main Card -->
        <div class="request-card">
            <!-- Welcome Section -->
            <div class="welcome-section">
                <h2 class="welcome-title">مرحباً <?php echo htmlspecialchars($ticket['customer_name'], ENT_QUOTES, 'UTF-8'); ?></h2>
                <p class="welcome-message">يمكنك طلب سيارتك الآن وسنقوم بتجهيزها فوراً</p>
            </div>

            <!-- Alerts -->
            <?php if ($statusMessage): ?>
                <div class="alert success">
                    <span class="material-icons alert-icon">check_circle</span>
                    <div class="alert-content"><?php echo htmlspecialchars($statusMessage, ENT_QUOTES, 'UTF-8'); ?></div>
                </div>
            <?php endif; ?>

            <?php if ($errorMessage): ?>
                <div class="alert error">
                    <span class="material-icons alert-icon">error</span>
                    <div class="alert-content"><?php echo htmlspecialchars($errorMessage, ENT_QUOTES, 'UTF-8'); ?></div>
                </div>
            <?php endif; ?>

            <!-- Vehicle Info Grid -->
            <div class="info-grid">
                <div class="info-item">
                    <div class="info-icon">
                        <span class="material-icons">info</span>
                    </div>
                    <div class="info-content">
                        <div class="info-label">حالة سيارتك</div>
                        <div class="info-value">
                            <span class="status-badge <?php echo $ticket['status']; ?>">
                                <span class="material-icons" style="font-size: 18px;">
                                    <?php 
                                    $icons = ['parked' => 'local_parking', 'requested' => 'schedule', 'returned' => 'check_circle'];
                                    echo $icons[$ticket['status']] ?? 'local_parking';
                                    ?>
                                </span>
                                <?php echo htmlspecialchars(statusLabel($ticket['status']), ENT_QUOTES, 'UTF-8'); ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Action Button -->
            <div class="action-section">
                <form method="post">
                    <button type="submit" class="request-button" <?php echo ($isRequested || $isReturned) ? 'disabled' : ''; ?>>
                        <span class="material-icons">
                            <?php echo ($isRequested || $isReturned) ? 'check_circle' : 'notifications_active'; ?>
                        </span>
                        <span>
                            <?php 
                            if ($isReturned) {
                                echo 'تم تسليم السيارة';
                            } elseif ($isRequested) {
                                echo 'تم إرسال الطلب - جاري التجهيز';
                            } else {
                                echo 'أطلب سيارتي الآن';
                            }
                            ?>
                        </span>
                    </button>
                </form>
            </div>
        </div>
    </div>
</body>
</html>
<?php

function statusLabel(string $status): string
{
    $labels = [
        'parked' => 'موقف',
        'requested' => 'قيد التجهيز',
        'returned' => 'مستلم',
    ];

    return $labels[$status] ?? $status;
}
