<?php
// Enable error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<h2>Database Connection Test</h2>";

// Load environment variables
$envFile = dirname(__DIR__) . '/.env';
echo "<p><strong>Environment file:</strong> " . ($envFile) . "</p>";
echo "<p><strong>File exists:</strong> " . (file_exists($envFile) ? 'YES' : 'NO') . "</p>";

if (!file_exists($envFile)) {
    die("<p style='color:red'>ERROR: .env file not found!</p>");
}

// Parse .env file
$envContent = file_get_contents($envFile);
$config = [];
foreach (explode("\n", $envContent) as $line) {
    $line = trim($line);
    if ($line === '' || strpos($line, '#') === 0 || strpos($line, '=') === false) {
        continue;
    }
    [$key, $value] = array_map('trim', explode('=', $line, 2));
    $config[$key] = trim($value, " \"'\t");
}

echo "<h3>Database Configuration:</h3>";
echo "<pre>";
echo "DB_DRIVER: " . ($config['DB_DRIVER'] ?? 'NOT SET') . "\n";
echo "DB_HOST: " . ($config['DB_HOST'] ?? 'NOT SET') . "\n";
echo "DB_PORT: " . ($config['DB_PORT'] ?? 'NOT SET') . "\n";
echo "DB_NAME: " . ($config['DB_NAME'] ?? 'NOT SET') . "\n";
echo "DB_USER: " . ($config['DB_USER'] ?? 'NOT SET') . "\n";
echo "DB_PASSWORD: " . (isset($config['DB_PASSWORD']) ? str_repeat('*', strlen($config['DB_PASSWORD'])) : 'NOT SET') . "\n";
echo "</pre>";

// Check PDO MySQL extension
echo "<h3>PHP Extensions:</h3>";
echo "<p><strong>PDO:</strong> " . (extension_loaded('pdo') ? 'YES ✓' : 'NO ✗') . "</p>";
echo "<p><strong>PDO MySQL:</strong> " . (extension_loaded('pdo_mysql') ? 'YES ✓' : 'NO ✗') . "</p>";

if (!extension_loaded('pdo_mysql')) {
    die("<p style='color:red'>ERROR: PDO MySQL extension is not installed!</p>");
}

// Test database connection
echo "<h3>Connection Test:</h3>";

try {
    $dsn = sprintf(
        'mysql:host=%s;port=%s;dbname=%s;charset=%s',
        $config['DB_HOST'] ?? 'localhost',
        $config['DB_PORT'] ?? '3306',
        $config['DB_NAME'] ?? '',
        $config['DB_CHARSET'] ?? 'utf8mb4'
    );
    
    echo "<p><strong>DSN:</strong> " . str_replace($config['DB_PASSWORD'] ?? '', '****', $dsn) . "</p>";
    
    $pdo = new PDO(
        $dsn,
        $config['DB_USER'] ?? '',
        $config['DB_PASSWORD'] ?? '',
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
    
    echo "<p style='color:green; font-weight:bold'>✓ CONNECTION SUCCESSFUL!</p>";
    
    // Test a simple query
    $stmt = $pdo->query('SELECT DATABASE() as db, VERSION() as version');
    $result = $stmt->fetch();
    
    echo "<p><strong>Connected to database:</strong> " . $result['db'] . "</p>";
    echo "<p><strong>MySQL version:</strong> " . $result['version'] . "</p>";
    
    // Check if tables exist
    $stmt = $pdo->query('SHOW TABLES');
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    echo "<h3>Tables in database:</h3>";
    if (empty($tables)) {
        echo "<p style='color:orange'>No tables found. Database needs to be migrated.</p>";
    } else {
        echo "<ul>";
        foreach ($tables as $table) {
            echo "<li>" . htmlspecialchars($table) . "</li>";
        }
        echo "</ul>";
    }
    
} catch (PDOException $e) {
    echo "<p style='color:red'><strong>✗ CONNECTION FAILED!</strong></p>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p><strong>Error Code:</strong> " . $e->getCode() . "</p>";
}

echo "<hr>";
echo "<p><a href='index.php'>← Back to Home</a></p>";
