<?php

namespace Parking;

class Auth
{
    private const SESSION_KEY = 'admin_id';
    private const SESSION_USERNAME = 'admin_username';
    
    private Database $database;

    public function __construct(Database $database)
    {
        $this->database = $database;
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }

    public function login(string $username, string $password): bool
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare('SELECT id, username, password_hash, full_name FROM admins WHERE username = :username');
        $stmt->execute([':username' => $username]);
        $admin = $stmt->fetch();

        if (!$admin) {
            return false;
        }

        if (!password_verify($password, $admin['password_hash'])) {
            return false;
        }

        // Update last login
        $updateStmt = $pdo->prepare('UPDATE admins SET last_login_at = NOW() WHERE id = :id');
        $updateStmt->execute([':id' => $admin['id']]);

        // Set session - regenerate id to prevent fixation attacks
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        session_regenerate_id(true);
        $_SESSION[self::SESSION_KEY] = $admin['id'];
        $_SESSION[self::SESSION_USERNAME] = $admin['username'];
        $_SESSION['admin_full_name'] = $admin['full_name'];

        return true;
    }

    public function logout(): void
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION = [];
        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000, $params['path'] ?: '/', $params['domain'] ?: '', !empty($params['secure']), !empty($params['httponly']));
        }
        session_destroy();
    }

    public function isLoggedIn(): bool
    {
        return isset($_SESSION[self::SESSION_KEY]);
    }

    public function getAdminId(): ?int
    {
        return $_SESSION[self::SESSION_KEY] ?? null;
    }

    public function getAdminUsername(): ?string
    {
        return $_SESSION[self::SESSION_USERNAME] ?? null;
    }

    public function getAdminFullName(): ?string
    {
        return $_SESSION['admin_full_name'] ?? null;
    }

    public function requireAuth(): void
    {
        if (!$this->isLoggedIn()) {
            // Use relative path so redirects work when the app is served from a subfolder
            header('Location: login.php');
            exit;
        }
    }
}
