<?php

namespace Parking;

use PDO;

class CoordinatorRepository
{
    private Database $database;

    public function __construct(Database $database)
    {
        $this->database = $database;
    }

    public function getAllCoordinators(): array
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->query(
            'SELECT c.*, e.name AS event_name, cl.name AS client_name
             FROM coordinators c
             LEFT JOIN events e ON c.event_id = e.id
             LEFT JOIN clients cl ON e.client_id = cl.id
             ORDER BY c.created_at DESC'
        );
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function getCoordinatorsByEvent(int $eventId): array
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare(
            'SELECT c.*, e.name AS event_name, cl.name AS client_name
             FROM coordinators c
             LEFT JOIN events e ON c.event_id = e.id
             LEFT JOIN clients cl ON e.client_id = cl.id
             WHERE c.event_id = :event_id
             ORDER BY c.created_at DESC'
        );
        $stmt->execute([':event_id' => $eventId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function getCoordinatorById(int $id): ?array
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare(
            'SELECT c.*, e.name AS event_name, cl.name AS client_name
             FROM coordinators c
             LEFT JOIN events e ON c.event_id = e.id
             LEFT JOIN clients cl ON e.client_id = cl.id
             WHERE c.id = :id'
        );
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function getCoordinatorByUsername(string $username): ?array
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare('SELECT * FROM coordinators WHERE username = :username');
        $stmt->execute([':username' => $username]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function createCoordinator(array $data): bool
    {
        $pdo = $this->database->connection();

        $checkStmt = $pdo->prepare('SELECT COUNT(*) FROM coordinators WHERE username = :username');
        $checkStmt->execute([':username' => $data['username']]);
        if ((int)$checkStmt->fetchColumn() > 0) {
            return false;
        }

        $stmt = $pdo->prepare(
            'INSERT INTO coordinators (event_id, full_name, username, password_hash, phone, email, is_active)
             VALUES (:event_id, :full_name, :username, :password_hash, :phone, :email, 1)'
        );

        return (bool) $stmt->execute([
            ':event_id' => $data['event_id'] ?? null,
            ':full_name' => $data['full_name'],
            ':username' => $data['username'],
            ':password_hash' => password_hash($data['password'], PASSWORD_BCRYPT),
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
        ]);
    }

    public function updateCoordinator(int $id, array $data): bool
    {
        $pdo = $this->database->connection();

        $checkStmt = $pdo->prepare('SELECT COUNT(*) FROM coordinators WHERE username = :username AND id != :id');
        $checkStmt->execute([':username' => $data['username'], ':id' => $id]);
        if ((int)$checkStmt->fetchColumn() > 0) {
            return false;
        }

        $fields = [
            'event_id = :event_id',
            'full_name = :full_name',
            'username = :username',
            'phone = :phone',
            'email = :email',
            'updated_at = CURRENT_TIMESTAMP',
        ];
        $params = [
            ':id' => $id,
            ':event_id' => $data['event_id'] ?? null,
            ':full_name' => $data['full_name'],
            ':username' => $data['username'],
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
        ];

        if (!empty($data['password'])) {
            $fields[] = 'password_hash = :password_hash';
            $params[':password_hash'] = password_hash($data['password'], PASSWORD_BCRYPT);
        }

        $sql = 'UPDATE coordinators SET ' . implode(', ', $fields) . ' WHERE id = :id';
        $stmt = $pdo->prepare($sql);
        return (bool) $stmt->execute($params);
    }

    public function toggleCoordinatorStatus(int $id): bool
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare(
            'UPDATE coordinators
             SET is_active = CASE WHEN is_active = 1 THEN 0 ELSE 1 END,
                 updated_at = CURRENT_TIMESTAMP
             WHERE id = :id'
        );
        return (bool) $stmt->execute([':id' => $id]);
    }

    public function deleteCoordinator(int $id): bool
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare('DELETE FROM coordinators WHERE id = :id');
        return (bool) $stmt->execute([':id' => $id]);
    }

    public function verifyLogin(string $username, string $password)
    {
        $coordinator = $this->getCoordinatorByUsername($username);
        if ($coordinator && isset($coordinator['password_hash']) && password_verify($password, $coordinator['password_hash'])) {
            if (empty($coordinator['is_active'])) {
                return false; // Coordinator account is disabled
            }
            return $coordinator;
        }
        return false;
    }

    public function getCoordinatorStats(int $coordinatorId): ?array
    {
        $pdo = $this->database->connection();
        $coordinator = $this->getCoordinatorById($coordinatorId);
        if (!$coordinator) {
            return null;
        }
        $stmt = $pdo->prepare(
            'SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = "parked" THEN 1 ELSE 0 END) as parked,
                SUM(CASE WHEN status = "requested" THEN 1 ELSE 0 END) as requested,
                SUM(CASE WHEN status = "returned" THEN 1 ELSE 0 END) as returned
             FROM tickets
             WHERE coordinator_id = :cid'
        );
        $stmt->execute([':cid' => $coordinatorId]);
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: ['total' => 0, 'parked' => 0, 'requested' => 0, 'returned' => 0];
    }
}

