<?php

namespace Parking;

use PDO;
use PDOException;

class Database
{
    private Config $config;
    private ?PDO $pdo = null;

    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    public function connection(): PDO
    {
        if ($this->pdo instanceof PDO) {
            return $this->pdo;
        }

        $driver = $this->config->get('DB_DRIVER', 'mysql');

        if ($driver === 'sqlite') {
            return $this->connectSqlite();
        } else {
            return $this->connectMysql();
        }
    }

    private function connectSqlite(): PDO
    {
        // Check available SQLite methods
        $pdoSqlite = extension_loaded('pdo_sqlite');
        $sqlite3 = extension_loaded('sqlite3');
        $sqlite = extension_loaded('sqlite');

        if (!($pdoSqlite || $sqlite3 || $sqlite)) {
            throw new PDOException(
                'No SQLite extension found. ' .
                'Your hosting needs one of: pdo_sqlite, sqlite3 (from DB_Sqlite_Tools), or sqlite extension. ' .
                'Contact your hosting provider to enable SQLite support.'
            );
        }

        $dbPath = $this->config->get('DB_PATH', dirname(__DIR__) . '/data/parking.db');
        $absolutePath = $this->normalizePath($dbPath);
        $this->ensureDirectory($absolutePath);

        $dsn = 'sqlite:' . $absolutePath;

        try {
            $this->pdo = new PDO($dsn, null, null, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
            
            // Enable foreign key constraints for SQLite
            $this->pdo->exec('PRAGMA foreign_keys = ON');
            
            // Create database file if it doesn't exist
            if (!file_exists($absolutePath)) {
                touch($absolutePath);
                chmod($absolutePath, 0666);
            }
        } catch (PDOException $e) {
            throw new PDOException(
                'Unable to connect to SQLite database at ' . $absolutePath . ': ' . $e->getMessage() .
                ' (Using: ' . ($pdoSqlite ? 'pdo_sqlite' : ($sqlite3 ? 'sqlite3/DB_Sqlite_Tools' : 'sqlite')) . ')',
                (int) $e->getCode(),
                $e
            );
        }

        return $this->pdo;
    }

    private function connectMysql(): PDO
    {
        // Check if pdo_mysql is available
        if (!extension_loaded('pdo_mysql')) {
            throw new PDOException(
                'PDO MySQL extension (pdo_mysql) is not installed. ' .
                'Contact your hosting provider to enable it in php.ini.'
            );
        }

        $host = $this->config->get('DB_HOST', 'localhost');
        $port = $this->config->get('DB_PORT', 3306);
        $database = $this->config->get('DB_NAME', 'parking');
        $user = $this->config->get('DB_USER', 'root');
        $password = $this->config->get('DB_PASSWORD', '');
        $charset = $this->config->get('DB_CHARSET', 'utf8mb4');

        // Include database name in DSN for MySQL
        $dsn = "mysql:host={$host};port={$port};dbname={$database};charset={$charset}";

        try {
            $this->pdo = new PDO($dsn, $user, $password, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        } catch (PDOException $e) {
            throw new PDOException(
                'Unable to connect to MySQL database ' . $database . ' at ' . $host . ':' . $port . ': ' . $e->getMessage(),
                (int) $e->getCode(),
                $e
            );
        }

        return $this->pdo;
    }

    private function normalizePath(string $path): string
    {
        $isAbsoluteWindows = preg_match('/^[A-Za-z]:\\\\/', $path) === 1;
        // PHP 7.4 compatibility: replace str_starts_with() with strpos
        $isAbsoluteUnix = strpos($path, '/') === 0;

        if ($isAbsoluteWindows || $isAbsoluteUnix) {
            return $path;
        }

        $base = realpath(dirname(__DIR__));
        if ($base === false) {
            $base = dirname(__DIR__);
        }

        return rtrim($base, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($path, './\\/');
    }

    private function ensureDirectory(string $path): void
    {
        $dir = dirname($path);
        if (!is_dir($dir)) {
            $result = @mkdir($dir, 0775, true);
            if (!$result) {
                // Try with less restrictive permissions on shared hosting
                @mkdir($dir, 0777, true);
            }
        }

        // Make sure the directory is writable
        if (!is_writable($dir)) {
            @chmod($dir, 0777);
        }
    }
}
