<?php

namespace Parking;

class EventRepository
{
    private $database;

    public function __construct(Database $database)
    {
        $this->database = $database;
    }

    /**
     * Get active event (only one event can be active at a time)
     */
    public function getActiveEvent()
    {
        $pdo = $this->database->connection();
            try {
                if (!$this->supportsEventWithClient()) {
                    // Schema does not support events with client/is_active; return no active event
                    return null;
                }

                $stmt = $pdo->prepare("
                    SELECT e.*, c.name as client_name, c.logo as client_logo, 
                           c.contact_person, c.contact_phone, c.contact_email
                    FROM events e
                    JOIN clients c ON e.client_id = c.id
                    WHERE e.is_active = 1
                    LIMIT 1
                ");
                $stmt->execute();
                return $stmt->fetch();
            } catch (\Throwable $e) {
                // Fail safe: no active event
                return null;
            }
    }

    /**
     * Get all events with client info
     */
    public function getAllEvents()
    {
        $pdo = $this->database->connection();
            try {
                if (!$this->supportsEventWithClient()) {
                    return [];
                }
                $stmt = $pdo->query("
                    SELECT e.*, c.name as client_name, c.logo as client_logo
                    FROM events e
                    JOIN clients c ON e.client_id = c.id
                    ORDER BY e.created_at DESC
                ");
                return $stmt->fetchAll();
            } catch (\Throwable $e) {
                return [];
            }
    }

    /**
     * Get events by client ID
     */
    public function getEventsByClient($clientId)
    {
        $pdo = $this->database->connection();
            try {
                if (!$this->supportsEventWithClient()) {
                    return [];
                }
                $stmt = $pdo->prepare("
                    SELECT * FROM events 
                    WHERE client_id = :client_id 
                    ORDER BY created_at DESC
                ");
                $stmt->execute(['client_id' => $clientId]);
                return $stmt->fetchAll();
            } catch (\Throwable $e) {
                return [];
            }
    }

    /**
     * Get event by ID
     */
    public function getEventById($id)
    {
        $pdo = $this->database->connection();
            try {
                if (!$this->supportsEventWithClient()) {
                    return null;
                }
                $stmt = $pdo->prepare("
                    SELECT e.*, c.name as client_name, c.logo as client_logo,
                           c.contact_person, c.contact_phone, c.contact_email
                    FROM events e
                    JOIN clients c ON e.client_id = c.id
                    WHERE e.id = :id
                ");
                $stmt->execute(['id' => $id]);
                return $stmt->fetch();
            } catch (\Throwable $e) {
                return null;
            }
    }

    /**
     * Create new event
     */
    public function createEvent($data)
    {
        $pdo = $this->database->connection();
            if (!$this->supportsEventWithClient()) {
                return false;
            }
            $stmt = $pdo->prepare("
                INSERT INTO events (client_id, name, logo, event_date, event_location, description)
                VALUES (:client_id, :name, :logo, :event_date, :event_location, :description)
            ");
            return $stmt->execute([
                'client_id' => $data['client_id'],
                'name' => $data['name'],
                'logo' => $data['logo'] ?? null,
                'event_date' => $data['event_date'] ?? null,
                'event_location' => $data['event_location'] ?? null,
                'description' => $data['description'] ?? null
            ]);
    }

    /**
     * Update event
     */
    public function updateEvent($id, $data)
    {
        $pdo = $this->database->connection();
            if (!$this->supportsEventWithClient()) {
                return false;
            }
            $stmt = $pdo->prepare("
                UPDATE events 
                SET name = :name,
                    logo = :logo,
                    event_date = :event_date,
                    event_location = :event_location,
                    description = :description,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = :id
            ");
            return $stmt->execute([
                'id' => $id,
                'name' => $data['name'],
                'logo' => $data['logo'] ?? null,
                'event_date' => $data['event_date'] ?? null,
                'event_location' => $data['event_location'] ?? null,
                'description' => $data['description'] ?? null
            ]);
    }

    /**
     * Activate event (deactivate all others first)
     */
    public function activateEvent($id)
    {
        $pdo = $this->database->connection();
            if (!$this->supportsEventWithClient()) {
                return false;
            }
            $pdo->beginTransaction();
            try {
                $pdo->exec("UPDATE events SET is_active = 0");
                $stmt = $pdo->prepare("UPDATE events SET is_active = 1 WHERE id = :id");
                $stmt->execute(['id' => $id]);
                $pdo->commit();
                return true;
            } catch (\Exception $e) {
                $pdo->rollBack();
                return false;
            }
    }

    /**
     * Deactivate event
     */
    public function deactivateEvent($id)
    {
        $pdo = $this->database->connection();
            if (!$this->supportsEventWithClient()) {
                return false;
            }
            $stmt = $pdo->prepare("UPDATE events SET is_active = 0 WHERE id = :id");
            return $stmt->execute(['id' => $id]);
    }

    /**
     * Delete event
     */
    public function deleteEvent($id)
    {
        if (!$this->supportsEventWithClient()) {
            return false;
        }
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare("DELETE FROM events WHERE id = :id");
        return $stmt->execute(['id' => $id]);
    }

    /**
     * Get active event count
     */
    public function getActiveEventCount()
    {
        try {
            if (!$this->supportsEventWithClient()) {
                return 0;
            }
            $pdo = $this->database->connection();
            $stmt = $pdo->query("SELECT COUNT(*) FROM events WHERE is_active = 1");
            return (int) $stmt->fetchColumn();
        } catch (\Throwable $e) {
            return 0;
        }
    }

    private function supportsEventWithClient(): bool
    {
        try {
            $pdo = $this->database->connection();
            $hasEvents = $this->tableHasColumns('events', ['client_id', 'is_active']);
            $hasClients = $this->tableExists('clients');
            return $hasEvents && $hasClients;
        } catch (\Throwable $e) {
            return false;
        }
    }

    private function tableExists(string $table): bool
    {
        $pdo = $this->database->connection();
        $driver = $pdo->getAttribute(\PDO::ATTR_DRIVER_NAME);
        if ($driver === 'mysql') {
            $stmt = $pdo->prepare("SELECT 1 FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t LIMIT 1");
            $stmt->execute([':t' => $table]);
            return (bool) $stmt->fetchColumn();
        }
        $stmt = $pdo->prepare("SELECT name FROM sqlite_master WHERE type = 'table' AND name = :t LIMIT 1");
        $stmt->execute([':t' => $table]);
        return (bool) $stmt->fetchColumn();
    }

    private function tableHasColumns(string $table, array $columns): bool
    {
        $pdo = $this->database->connection();
        $driver = $pdo->getAttribute(\PDO::ATTR_DRIVER_NAME);
        $existing = [];
        if ($driver === 'mysql') {
            $stmt = $pdo->prepare("SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t");
            $stmt->execute([':t' => $table]);
            $existing = array_map('strtolower', array_map('strval', $stmt->fetchAll(\PDO::FETCH_COLUMN)) ?: []);
        } else {
            $stmt = $pdo->prepare("PRAGMA table_info($table)");
            $stmt->execute();
            $rows = $stmt->fetchAll();
            foreach ($rows as $row) {
                if (isset($row['name'])) {
                    $existing[] = strtolower($row['name']);
                } elseif (isset($row[1])) {
                    $existing[] = strtolower($row[1]);
                }
            }
        }
        foreach ($columns as $c) {
            if (!in_array(strtolower($c), $existing, true)) {
                return false;
            }
        }
        return true;
    }
}
