<?php

namespace Parking;

use DateTimeImmutable;
use InvalidArgumentException;
use PDO;

class ParkingRepository
{
    private PDO $db;
    private ?Notifier $notifier;

    public function __construct(Database $database, ?Notifier $notifier = null)
    {
        $this->db = $database->connection();
        $this->notifier = $notifier;
    }

    public function registerVehicle(string $customerName, string $phoneNumber, string $licensePlate, string $valetName, ?int $eventId = null, ?int $coordinatorId = null, array $photos = []): int
    {
        $photosJson = !empty($photos) ? json_encode($photos) : null;
        
        $stmt = $this->db->prepare('INSERT INTO tickets (customer_name, phone_number, license_plate, valet_name, event_id, coordinator_id, photos, status, created_at, updated_at) VALUES (:customer_name, :phone_number, :license_plate, :valet_name, :event_id, :coordinator_id, :photos, :status, :created_at, :updated_at)');
        $now = $this->now();
        $stmt->execute([
            ':customer_name' => $customerName,
            ':phone_number' => $phoneNumber,
            ':license_plate' => $licensePlate,
            ':valet_name' => $valetName,
            ':event_id' => $eventId,
            ':coordinator_id' => $coordinatorId,
            ':photos' => $photosJson,
            ':status' => 'parked',
            ':created_at' => $now,
            ':updated_at' => $now,
        ]);

        return (int) $this->db->lastInsertId();
    }

    public function markRequested(int $ticketId): void
    {
        $stmt = $this->db->prepare('UPDATE tickets SET status = :status, requested_at = COALESCE(requested_at, :requested_at), updated_at = :updated_at WHERE id = :id AND status != :status_returned');
        $stmt->execute([
            ':status' => 'requested',
            ':requested_at' => $this->now(),
            ':updated_at' => $this->now(),
            ':status_returned' => 'returned',
            ':id' => $ticketId,
        ]);
    }

    public function markReturned(int $ticketId): void
    {
        // Get ticket info before updating
        $ticket = $this->findById($ticketId);
        
        $stmt = $this->db->prepare('UPDATE tickets SET status = :status, returned_at = COALESCE(returned_at, :returned_at), updated_at = :updated_at WHERE id = :id');
        $stmt->execute([
            ':status' => 'returned',
            ':returned_at' => $this->now(),
            ':updated_at' => $this->now(),
            ':id' => $ticketId,
        ]);
        
        // Send SMS notification if notifier is available
        if ($this->notifier && $ticket) {
            try {
                $message = "عزيزي {$ticket['customer_name']},\n\n";
                $message .= "سيارتك جاهزة للاستلام! ✅\n\n";
                $message .= "شكراً لاستخدام خدماتنا.\n";
                $message .= "Gulf Car Parking 🚗";
                
                $this->notifier->sendMessage($ticket['phone_number'], $message);
            } catch (\Exception $e) {
                // Log error but don't fail the operation
                error_log("Failed to send SMS notification: " . $e->getMessage());
            }
        }
    }

    public function markParked(int $ticketId): void
    {
        $stmt = $this->db->prepare('UPDATE tickets SET status = :status, requested_at = NULL, returned_at = NULL, updated_at = :updated_at WHERE id = :id');
        $stmt->execute([
            ':status' => 'parked',
            ':updated_at' => $this->now(),
            ':id' => $ticketId,
        ]);
    }

    public function getActiveTickets(?int $eventId = null, ?int $coordinatorId = null): array
    {
        if ($coordinatorId !== null) {
            // Coordinator: show only their tickets
            $stmt = $this->db->prepare('SELECT t.*, c.full_name as coordinator_name FROM tickets t LEFT JOIN coordinators c ON t.coordinator_id = c.id WHERE t.status IN ("parked", "requested") AND t.coordinator_id = :coordinator_id ORDER BY t.created_at DESC');
            $stmt->execute([':coordinator_id' => $coordinatorId]);
        } elseif ($eventId !== null) {
            // Admin with event filter
            $stmt = $this->db->prepare('SELECT t.*, c.full_name as coordinator_name FROM tickets t LEFT JOIN coordinators c ON t.coordinator_id = c.id WHERE t.status IN ("parked", "requested") AND t.event_id = :event_id ORDER BY t.created_at DESC');
            $stmt->execute([':event_id' => $eventId]);
        } else {
            // Admin: show all
            $stmt = $this->db->query('SELECT t.*, c.full_name as coordinator_name FROM tickets t LEFT JOIN coordinators c ON t.coordinator_id = c.id WHERE t.status IN ("parked", "requested") ORDER BY t.created_at DESC');
        }
        return $stmt->fetchAll();
    }

    public function getAllTickets(?int $eventId = null, ?int $coordinatorId = null): array
    {
        if ($coordinatorId !== null) {
            // Coordinator: show only their tickets
            $stmt = $this->db->prepare('SELECT t.*, c.full_name as coordinator_name FROM tickets t LEFT JOIN coordinators c ON t.coordinator_id = c.id WHERE t.coordinator_id = :coordinator_id ORDER BY t.created_at DESC');
            $stmt->execute([':coordinator_id' => $coordinatorId]);
        } elseif ($eventId !== null) {
            // Admin with event filter
            $stmt = $this->db->prepare('SELECT t.*, c.full_name as coordinator_name FROM tickets t LEFT JOIN coordinators c ON t.coordinator_id = c.id WHERE t.event_id = :event_id ORDER BY t.created_at DESC');
            $stmt->execute([':event_id' => $eventId]);
        } else {
            // Admin: show all
            $stmt = $this->db->query('SELECT t.*, c.full_name as coordinator_name FROM tickets t LEFT JOIN coordinators c ON t.coordinator_id = c.id ORDER BY t.created_at DESC');
        }
        return $stmt->fetchAll();
    }

    public function findById(int $ticketId): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM tickets WHERE id = :id LIMIT 1');
        $stmt->execute([':id' => $ticketId]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function findByToken(string $token): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM tickets WHERE request_token = :token LIMIT 1');
        $stmt->execute([':token' => $token]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function storeRequestToken(int $ticketId, string $token, string $requestUrl): void
    {
        $stmt = $this->db->prepare('UPDATE tickets SET request_token = :token, request_url = :url, updated_at = :updated_at WHERE id = :id');
        $stmt->execute([
            ':token' => $token,
            ':url' => $requestUrl,
            ':updated_at' => $this->now(),
            ':id' => $ticketId,
        ]);
    }

    public function setStatus(int $ticketId, string $status): ?array
    {
        $status = strtolower($status);

        switch ($status) {
            case 'parked':
                $this->markParked($ticketId);
                break;
            case 'requested':
                $this->markRequested($ticketId);
                break;
            case 'returned':
                $this->markReturned($ticketId);
                break;
            default:
                throw new InvalidArgumentException('حالة غير معروفة: ' . $status);
        }

        return $this->findById($ticketId);
    }

    private function now(): string
    {
        return (new DateTimeImmutable())->format('Y-m-d H:i:s');
    }
}
